package ru.yandex.solomon.codec.histogram;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListViewIterator;
import ru.yandex.solomon.model.type.Histogram;

/*

Benchmark                                            (pointCount)  Mode  Cnt   Score    Error  Units
HistogramEncoderJmhBenchmark.compressAsDoubles               1000  avgt   10  28.943 ±  2.698  ms/op
HistogramEncoderJmhBenchmark.compressAsHistograms            1000  avgt   10  18.470 ±  1.905  ms/op
HistogramEncoderJmhBenchmark.decompressAsHistograms          1000  avgt   10  25.781 ±  7.168  ms/op
HistogramEncoderJmhBenchmark.decompressDoubles               1000  avgt   10  24.265 ±  7.958  ms/op
HistogramEncoderJmhBenchmark.doublesBaseline                 1000  avgt   10   0.017 ±  0.002  ms/op
HistogramEncoderJmhBenchmark.histogramBaseline               1000  avgt   10   0.002 ±  0.001  ms/op

*/

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, batchSize = 50)
@Warmup(iterations = 5, batchSize = 50)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class HistogramEncoderJmhBenchmark {
    private final double[] bounds = {5, 10, 50, 100, 300, 500, 1000, 5_000, 15_000, 30_000, Histograms.INF_BOUND };
    @Param({"1000"})
    private int pointCount;
    private AggrGraphDataArrayList source;
    private BitBuf compressedHistograms;
    private BitBuf[] compressedDoubles;

    public static void main(String[] args) throws Throwable {
        Options opt = new OptionsBuilder()
                .include(HistogramEncoderJmhBenchmark.class.getName())
                .detectJvmArgs()
                .build();

        new Runner(opt).run();
    }

    @Setup()
    public void setUp() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask();
        source = new AggrGraphDataArrayList(mask, pointCount);
        long stepMillis = TimeUnit.SECONDS.toMillis(15L);
        long now = System.currentTimeMillis();
        AggrPointData point = new AggrPointData();
        for (int index = 0; index < pointCount; index++) {
            now += stepMillis;
            point.tsMillis = now;
            point.histogram = Histogram.newInstance().copyFrom(bounds, generateBuckets());
            source.addRecordData(mask, point);
        }

        compressedHistograms = compressAsHistograms();
        compressedDoubles = compressAsDoubles();
    }

    private long[] generateBuckets() {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        long[] buckets = new long[bounds.length];
        for (int index = 0; index < buckets.length; index++) {
            buckets[index] = random.nextLong(0, 10000);
        }
        return buckets;
    }

    @Benchmark
    public BitBuf compressAsHistograms() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask();
        TimeSeriesOutputStream stream = CompressStreamFactory.createOutputStream(MetricType.HIST, mask, pointCount);
        AggrPoint point = new AggrPoint();
        AggrGraphDataArrayListViewIterator it = source.iterator();
        while (it.next(point)) {
            stream.writePoint(mask, point);
        }
        return stream.getCompressedData();
    }

    @Benchmark
    public Object histogramBaseline() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask();
        return CompressStreamFactory.createOutputStream(MetricType.HIST, mask, pointCount);
    }

    @Benchmark
    public AggrGraphDataArrayList decompressAsHistograms() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask();
        TimeSeriesInputStream stream = CompressStreamFactory.createInputStream(MetricType.HIST, mask, compressedHistograms.duplicate());

        AggrPoint point = new AggrPoint();
        AggrGraphDataArrayList result = new AggrGraphDataArrayList(mask, pointCount);
        while (stream.hasNext()) {
            stream.readPoint(mask, point);
            result.addRecordData(mask, point);
        }
        return result;
    }

    @Benchmark
    public Object doublesBaseline() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        TimeSeriesOutputStream[] streams = new TimeSeriesOutputStream[bounds.length];
        for (int index = 0; index < bounds.length; index++) {
            streams[index] = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, mask, pointCount);
        }
        return streams;
    }

    @Benchmark
    public BitBuf[] compressAsDoubles() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        TimeSeriesOutputStream[] streams = new TimeSeriesOutputStream[bounds.length];
        for (int index = 0; index < bounds.length; index++) {
            streams[index] = CompressStreamFactory.createOutputStream(MetricType.DGAUGE, mask, pointCount);
        }

        AggrPoint point = new AggrPoint();
        AggrGraphDataArrayListViewIterator it = source.iterator();
        while (it.next(point)) {
            for (int index = 0; index < bounds.length; index++) {
                point.valueNum = point.histogram.value(index);
                streams[index].writePoint(mask, point);
            }
        }

        BitBuf[] results = new BitBuf[streams.length];
        for (int index = 0; index < streams.length; index++) {
            results[index] = streams[index].getCompressedData();
        }

        return results;
    }

    @Benchmark
    public AggrGraphDataArrayList decompressDoubles() {
        final int dMask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        final int hMask = StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask();
        TimeSeriesInputStream[] streams = new TimeSeriesInputStream[bounds.length];
        for (int index = 0; index < bounds.length; index++) {
            streams[index] = CompressStreamFactory.createInputStream(MetricType.DGAUGE, dMask, compressedDoubles[index].duplicate());
        }

        AggrGraphDataArrayList result = new AggrGraphDataArrayList(hMask, pointCount);
        AggrPoint point = new AggrPoint();
        while (streams[0].hasNext()) {
            long[] buckets = new long[bounds.length];
            for (int index = 0; index < buckets.length; index++) {
                streams[index].readPoint(dMask, point);
                buckets[index] = (long) point.valueNum;
            }
            point.histogram = Histogram.newInstance().copyFrom(bounds, buckets);
            result.addRecordData(hMask, point);
        }
        return result;
    }
}
