package ru.yandex.solomon.codec.histogram.log;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.model.type.LogHistogram;

/**
 * <pre>
 *     Benchmark                             (dataSetId)  Mode  Cnt     Score     Error  Units
 *     JmhLogHistogramBenchmark.bestGorilla            0  avgt    3    11.785 ±   0.034  ms/op
 *     JmhLogHistogramBenchmark.bestGorilla            1  avgt    3     6.583 ±   6.617  ms/op
 *     JmhLogHistogramBenchmark.bestGorilla            2  avgt    3     7.062 ±   1.349  ms/op
 *     JmhLogHistogramBenchmark.gzip                   0  avgt    3  1036.242 ± 443.997  ms/op
 *     JmhLogHistogramBenchmark.gzip                   1  avgt    3  1142.348 ± 714.582  ms/op
 *     JmhLogHistogramBenchmark.gzip                   2  avgt    3  1139.009 ± 449.786  ms/op
 *     JmhLogHistogramBenchmark.gzipBulk               0  avgt    3   253.350 ±  51.359  ms/op
 *     JmhLogHistogramBenchmark.gzipBulk               1  avgt    3   143.497 ±  11.963  ms/op
 *     JmhLogHistogramBenchmark.gzipBulk               2  avgt    3   112.001 ±   4.010  ms/op
 *     JmhLogHistogramBenchmark.protobuf               0  avgt    3    43.713 ±   1.838  ms/op
 *     JmhLogHistogramBenchmark.protobuf               1  avgt    3    24.461 ±   0.881  ms/op
 *     JmhLogHistogramBenchmark.protobuf               2  avgt    3    51.637 ± 125.818  ms/op
 *     JmhLogHistogramBenchmark.snappy                 0  avgt    3    72.510 ± 171.875  ms/op
 *     JmhLogHistogramBenchmark.snappy                 1  avgt    3    40.202 ±   1.602  ms/op
 *     JmhLogHistogramBenchmark.snappy                 2  avgt    3    47.773 ±   2.427  ms/op
 *     JmhLogHistogramBenchmark.snappyBulk             0  avgt    3    41.814 ±   3.009  ms/op
 *     JmhLogHistogramBenchmark.snappyBulk             1  avgt    3    26.649 ±  30.866  ms/op
 *     JmhLogHistogramBenchmark.snappyBulk             2  avgt    3    29.506 ±   1.453  ms/op
 * </pre>
 *
 * @author Vladimir Gordiychuk
 */
@Fork(value = 0)
@Measurement(iterations = 3, batchSize = 5)
@Warmup(iterations = 3, batchSize = 5)
@State(Scope.Thread)
@Threads(1)
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class JmhLogHistogramBenchmark {
    private static Encoder ENCODER_GORILLA = new EncoderGorilla();
    private static Encoder ENCODER_PROTOBUF = new EncoderAsIsProtobuf();
    private static Encoder ENCODER_SNAPPY = new EncoderAsIsProtobufSnappy();
    private static Encoder ENCODER_SNAPPY_BULK = new EncoderAsIsProtobufSnappyBulk();
    private static Encoder ENCODER_GZIP = new EncoderAsIsProtobufGzip();
    private static Encoder ENCODER_GZIP_BULK = new EncoderAsIsProtobufGzipBulk();

    @Param({"0", "1", "2"})
    private int dataSetId;
    private LogHistogram[] data;

    public static void main(String[] args) throws RunnerException, IOException {
        Options opt = new OptionsBuilder()
                .include(JmhLogHistogramBenchmark.class.getName())
                .detectJvmArgs()
                .build();

        new Runner(opt).run();
    }

    @Setup(Level.Trial)
    public void setUp() throws IOException {
        data = LogHistogramDataSet.getDataSet(dataSetId);
    }

    @Benchmark
    public byte[] bestGorilla() {
        return ENCODER_GORILLA.encode(data);
    }

    @Benchmark
    public byte[] protobuf() {
        return ENCODER_PROTOBUF.encode(data);
    }

    @Benchmark
    public byte[] snappy() {
        return ENCODER_SNAPPY.encode(data);
    }

    @Benchmark
    public byte[] snappyBulk() {
        return ENCODER_SNAPPY_BULK.encode(data);
    }

    @Benchmark
    public byte[] gzip() {
        return ENCODER_GZIP.encode(data);
    }

    @Benchmark
    public byte[] gzipBulk() {
        return ENCODER_GZIP_BULK.encode(data);
    }
}
