package ru.yandex.solomon.codec.histogram.log;

import java.io.IOException;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.profile.GCProfiler;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;
import org.xerial.snappy.Snappy;

import ru.yandex.misc.io.gzip.GzipUtils;
import ru.yandex.salmon.proto.StockpileCanonicalProto;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 0)
@Measurement(iterations = 5, time = 10, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 2, time = 10, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public class JmhPerPoitHistogramBenchmark {
    private final int LOG_HISTOGRAM_MASK = StockpileColumn.TS.mask() | StockpileColumn.LOG_HISTOGRAM.mask();
    private static LogHistogram[] DATA_SET;

    private int cursor;
    private TimeSeriesOutputStream outputStream;
    private AggrPoint point = new AggrPoint();

    @Setup(Level.Trial)
    public void setUp() throws IOException {
        point.setTsMillis(System.currentTimeMillis());
        outputStream = CompressStreamFactory.createOutputStream(MetricType.LOG_HISTOGRAM, LOG_HISTOGRAM_MASK);
        if (DATA_SET == null) {
            DATA_SET = IntStream.range(0, LogHistogramDataSet.COUNT_DATA_SETS)
                .mapToObj(LogHistogramDataSet::getDataSet)
                .flatMap(Stream::of)
                .toArray(LogHistogram[]::new);
        }
    }

    @Setup(Level.Iteration)
    public void iterationSetUp() throws IOException {
        cursor = 0;
        if (outputStream != null) {
            outputStream.close();
        }
        outputStream = CompressStreamFactory.createOutputStream(MetricType.LOG_HISTOGRAM, LOG_HISTOGRAM_MASK);
    }

    private void ensureData() {
        if (cursor >= DATA_SET.length) {
            cursor = 0;
        }
    }

    @Benchmark
    public BitBuf bestGorilla() {
        ensureData();
        point.logHistogram = nextHistogram();
        outputStream.writePoint(LOG_HISTOGRAM_MASK, point);
        return outputStream.getCompressedData();
    }

    private LogHistogram nextHistogram() {
        return DATA_SET[cursor++];
    }

    private StockpileCanonicalProto.LogHistogram nextProtobufHistogram() {
        return LogHistogramConverter.toProtobut(nextHistogram());
    }

    @Benchmark
    public byte[] protobuf() {
        ensureData();
        return nextProtobufHistogram().toByteArray();
    }

    @Benchmark
    public byte[] snappy() throws IOException {
        ensureData();
        return Snappy.compress(nextProtobufHistogram().toByteArray());
    }

    @Benchmark
    public byte[] gzip() {
        ensureData();
        return GzipUtils.zip(nextProtobufHistogram().toByteArray());
    }

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(JmhPerPoitHistogramBenchmark.class.getName())
            .addProfiler(GCProfiler.class)
            .detectJvmArgs()
            .build();

        new Runner(opt).run();
    }
}
