package ru.yandex.solomon.codec.histogram.log;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.IntSummaryStatistics;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * <pre>
 * Sorted by compression effective:
 * Name                               Total Size  Total size in bytes
 * EncoderAsIsProtobufGzipBulk             28 MB             30022659
 * EncoderGorilla                          38 MB             40091902
 * EncoderAsIsProtobufSnappyBulk           41 MB             43360026
 * EncoderSingleGorillaBucketState         45 MB             47643673
 * EncoderAsIsProtobufSnappy               76 MB             79873710
 * EncoderAsIsProtobufGzip                 91 MB             95626711
 * EncoderAsIs                            125 MB            131421841
 * EncoderAsIsProtobuf                    138 MB            145389813
 * </pre>
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
class LogHistogramCompressionBenchmark {
    private static final List<Encoder> ENCODERS = Arrays.asList(
            new EncoderAsIs(),
            new EncoderAsIsProtobuf(),
            new EncoderAsIsProtobufSnappy(),
            new EncoderAsIsProtobufSnappyBulk(),
            new EncoderAsIsProtobufGzip(),
            new EncoderAsIsProtobufGzipBulk(),
            new EncoderGorilla()
    );

    public static void main(String[] args) throws IOException {
        List<CompressionResult> results = new ArrayList<>();

        for (int index = 0; index < LogHistogramDataSet.COUNT_DATA_SETS; index++) {
            LogHistogram[] data = LogHistogramDataSet.getDataSet(index);

            IntSummaryStatistics bucketsStats = Stream.of(data)
                    .mapToInt(LogHistogram::countBucket)
                    .summaryStatistics();

            List<CompressionResult> iterationResult = new ArrayList<>();
            for (Encoder encoder : ENCODERS) {
                byte[] compressed = encoder.encode(data);

                CompressionResult encodeResult = new CompressionResult();
                encodeResult.fileName = index + ".json";
                encodeResult.encoder = encoder.getClass().getSimpleName();
                encodeResult.compressionLength = compressed.length;
                encodeResult.bitPerPoint = 1.0 * compressed.length * 8 / data.length;
                encodeResult.avgBucketsSize = (int) Math.round(bucketsStats.getAverage());
                encodeResult.maxBucketsSize = bucketsStats.getMax();
                encodeResult.minBucketsSize = bucketsStats.getMin();
                encodeResult.countHistograms = data.length;

                iterationResult.add(encodeResult);
            }

            printIterationResult(iterationResult);
            results.addAll(iterationResult);
        }
        printTop(results);
    }

    private static void printTop(List<CompressionResult> compressionResults) {
        Map<String, Long> encoderToTotalSize = compressionResults.stream()
                .collect(Collectors.toMap(o -> o.encoder, o -> o.compressionLength, (o, o2) -> o + o2));

        int nameLen = "Name".length();
        int totalSizeLen = "Total Size".length();
        int bytesLen = "Total size in bytes".length();

        for (Map.Entry<String, Long> entry : encoderToTotalSize.entrySet()) {
            nameLen = Math.max(entry.getKey().length(), nameLen);
            totalSizeLen = Math.max(DataSize.fromBytes(entry.getValue()).toStringShort().length(), totalSizeLen);
            bytesLen = Math.max(entry.getValue().toString().length(), bytesLen);
        }

        nameLen += 2;
        totalSizeLen += 2;
        bytesLen += 2;

        System.out.println("Sorted by compression effective:");
        System.out.printf("%-" + nameLen + "s", "Name");
        System.out.printf("%" + totalSizeLen + "s", "Total Size");
        System.out.printf("%" + bytesLen + "s", "Total size in bytes");
        System.out.println();

        int finalNameLen = nameLen;
        int finalTotalSizeLen = totalSizeLen;
        int finalBytesLen = bytesLen;
        encoderToTotalSize.entrySet()
                .stream()
                .sorted(Map.Entry.comparingByValue())
                .forEach(entry -> {
                    System.out.printf("%-" + finalNameLen + "s", entry.getKey());
                    System.out.printf("%" + finalTotalSizeLen + "s", DataSize.fromBytes(entry.getValue()).toStringShort());
                    System.out.printf("%" + finalBytesLen + "s", entry.getValue());
                    System.out.println();
                });
    }

    private static void printIterationResult(List<CompressionResult> iterationResult) {
        // determine name column length
        int nameLen = "Name".length();
        int fileLen = "File".length();
        int sizeLen = "Size".length();
        int bytesLen = "Bytes".length();
        int diffLen = "Diff".length();
        int pointBitLen = "Bits per point".length();
        int countLen = "Points".length();
        int maxBucketLen = "Max buckets".length();
        int minBucketLen = "Min buckets".length();
        int avgBucketLen = "Avg buckets".length();

        iterationResult.sort(Comparator.comparingLong(o -> o.compressionLength));

        long topCompressionLength = iterationResult.iterator().next().compressionLength;

        for (CompressionResult result : iterationResult) {
            nameLen = Math.max(result.encoder.length(), nameLen);
            fileLen = Math.max(result.fileName.length(), fileLen);
            bytesLen = Math.max(String.valueOf(result.compressionLength).length(), bytesLen);
            sizeLen = Math.max(DataSize.fromBytes(result.compressionLength).toStringShort().length(), sizeLen);
            pointBitLen = Math.max(String.format("%.1f", result.bitPerPoint).length(), pointBitLen);
            diffLen = Math.max(DataSize.fromBytes(result.compressionLength - topCompressionLength).toStringShort().length(), diffLen);
            countLen = Math.max(String.valueOf(result.countHistograms).length(), countLen);
            maxBucketLen = Math.max(String.valueOf(result.maxBucketsSize).length(), maxBucketLen);
            minBucketLen = Math.max(String.valueOf(result.minBucketsSize).length(), minBucketLen);
            avgBucketLen = Math.max(String.valueOf(result.avgBucketsSize).length(), avgBucketLen);
        }


        nameLen += 2;
        fileLen += 2;
        bytesLen += 2;
        sizeLen += 2;
        pointBitLen += 2;
        diffLen += 2;
        countLen += 2;
        maxBucketLen += 2;
        minBucketLen += 2;
        avgBucketLen += 2;


        System.out.printf("%-" + nameLen + "s", "Name");
        System.out.printf("%" + fileLen + "s", "File");
        System.out.printf("%" + sizeLen + "s", "Size");
        System.out.printf("%" + bytesLen + "s", "Bytes");
        System.out.printf("%" + pointBitLen + "s", "Bits per point");
        System.out.printf("%" + diffLen + "s", "Diff");
        System.out.printf("%" + countLen + "s", "Points");
        System.out.printf("%" + maxBucketLen + "s", "Max buckets");
        System.out.printf("%" + minBucketLen + "s", "Min buckets");
        System.out.printf("%" + avgBucketLen + "s", "Avg buckets");

        System.out.println();


        iterationResult.sort(Comparator.comparingLong(o -> o.compressionLength));
        for (CompressionResult result : iterationResult) {
            System.out.printf("%-" + nameLen + "s", result.encoder);
            System.out.printf("%" + fileLen + "s", result.fileName);
            System.out.printf("%" + sizeLen + "s", DataSize.fromBytes(result.compressionLength).toStringShort());
            System.out.printf("%" + bytesLen + "s", result.compressionLength);
            System.out.printf("%" + pointBitLen + ".1f", result.bitPerPoint);
            System.out.printf("%" + diffLen + "s", DataSize.fromBytes(result.compressionLength - topCompressionLength).toStringShort());
            System.out.printf("%" + countLen + "s", result.countHistograms);
            System.out.printf("%" + maxBucketLen + "s", result.maxBucketsSize);
            System.out.printf("%" + minBucketLen + "s", result.minBucketsSize);
            System.out.printf("%" + avgBucketLen + "s", result.avgBucketsSize);

            System.out.println();
        }
        System.out.println();
    }

    private static class CompressionResult {
        private String encoder;
        private String fileName;
        private long compressionLength;
        private double bitPerPoint;
        private int maxBucketsSize;
        private int minBucketsSize;
        private int avgBucketsSize;
        private int countHistograms;
    }
}
