package ru.yandex.solomon.codec.histogram.log;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import ru.yandex.misc.io.file.File2;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
final class LogHistogramDataSet {
    private LogHistogramDataSet() {
    }

    /**
     * Use persistent tmpDir to avoid download each test execution ~200mb
     */
    private static final Path CACHE_PATH = Paths.get(System.getProperty("java.io.tmpdir"), "solomon", "hgrams");

    /**
     * DataSet provided by GOLOVAN
     * https://st.yandex-team.ru/GOLOVAN-4671
     * View: http://sandbox.yandex-team.ru/resource/262437772/view
     */
    private static final String SANDBOX_RESOURCE = "https://proxy.sandbox.yandex-team.ru/262437772";
    static final int COUNT_DATA_SETS = 34;

    static LogHistogram[] getDataSet(int index) {
        try {
            Path path = downloadToCache(index);
            return readFromCache(path);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static Path downloadToCache(int index) throws IOException {
        Path path = CACHE_PATH.resolve(index+".json");
        if (Files.notExists(path)) {
            File2.wrap(CACHE_PATH.toFile()).mkdirs();
            Path tmp = Files.createTempFile("solomon", String.valueOf(index));
            try(InputStream is = openStreamToDataFile(index)) {
                Files.copy(is, tmp, StandardCopyOption.REPLACE_EXISTING);
                Files.move(tmp, path, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            }
        }

        return path;
    }

    private static LogHistogram[] readFromCache(Path path) throws IOException {
        try(InputStream is = Files.newInputStream(path, StandardOpenOption.READ)) {
            return LogHistogramParser.parse(is);
        }
    }

    private static URL urlToDataFile(int index) throws MalformedURLException {
        return new URL(SANDBOX_RESOURCE + "/" + index + ".json");
    }

    @WillNotClose
    private static InputStream openStreamToDataFile(int index) throws IOException {
        return urlToDataFile(index).openStream();
    }
}
