package ru.yandex.solomon.codec.histogram.log;

import java.io.IOException;
import java.io.InputStream;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;

import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.MutableLogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
class LogHistogramParser {
    private static JsonNode parseTree(@WillNotClose InputStream stream) {
        try {
            return new ObjectMapper().readTree(stream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static LogHistogram[] parse(@WillNotClose InputStream stream) {
        JsonNode node = parseTree(stream);
        if (!node.isArray()) {
            return new LogHistogram[0];
        }

        return IntStream.range(0, node.size())
            .mapToObj(node::get)
            .filter(histogramNode -> histogramNode.size() == 3 && histogramNode.get(0).isArray())
            .map(histogramNode -> {
                ArrayNode bucketsNode = (ArrayNode) histogramNode.get(0);
                double[] buckets =
                    IntStream.range(0, bucketsNode.size())
                        .mapToDouble(index -> bucketsNode.get(index).asDouble())
                        .toArray();

                int countZero = histogramNode.get(1).asInt();
                int startPower = histogramNode.get(2).asInt();

                //raw value need convert to histogram
                if (histogramNode.get(2).isNull()) {
                    MutableLogHistogram histogram = new MutableLogHistogram()
                        .addHistogram(LogHistogram.newBuilder()
                            .setStartPower(startPower)
                            .setCountZero(countZero)
                            .build());

                    for (double value : buckets) {
                        histogram.addValue(value);
                    }

                    return histogram.toImmutable();
                }

                return LogHistogram.newBuilder()
                    .setBuckets(buckets)
                    .setStartPower(startPower)
                    .setCountZero(countZero)
                    .build();
            }).toArray(LogHistogram[]::new);
    }

    public static LogHistogram[] read(String path) {
        try (InputStream stream = LogHistogramParser.class.getResourceAsStream(path)) {
            return parse(stream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
