package ru.yandex.solomon.codec.summary;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.LongSupplier;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class SummaryEncoderBenchmark {
    private static void testInt64(String name, List<SummaryInt64Snapshot> summaries) {
        final int mask = StockpileColumn.TS.mask() | StockpileColumn.ISUMMARY.mask();
        TimeSeriesOutputStream stream = CompressStreamFactory.createOutputStream(MetricType.ISUMMARY, mask, summaries.size());

        AggrPoint point = new AggrPoint();
        point.columnSet = mask;
        point.tsMillis = System.currentTimeMillis();
        for (SummaryInt64Snapshot summary : summaries) {
            point.summaryInt64 = summary;
            stream.writePoint(mask, point);
        }

        long lengthBits = stream.getCompressedData().readableBits();
        double b = 1.0 * lengthBits / summaries.size();
        double b2 = 1.0 * (lengthBits - 64) / (summaries.size() - 1);
        System.out.printf("%-20s: %.1f %.1f bits per point\n", name, b, b2);
    }

    private static void testDouble(String name, List<SummaryDoubleSnapshot> summaries) {
        final int mask = StockpileColumn.TS.mask() | StockpileColumn.DSUMMARY.mask();
        TimeSeriesOutputStream stream = CompressStreamFactory.createOutputStream(MetricType.DSUMMARY, mask);

        AggrPoint point = new AggrPoint();
        point.tsMillis = System.currentTimeMillis();
        for (SummaryDoubleSnapshot summary : summaries) {
            point.summaryDouble = summary;
            stream.writePoint(mask, point);
        }

        long lengthBits = stream.getCompressedData().readableBits();
        double b = 1.0 * lengthBits / summaries.size();
        double b2 = 1.0 * (lengthBits - 64) / (summaries.size() - 1);
        System.out.printf("%-20s: %.1f %.1f bits per point\n", name, b, b2);
    }

    private static long junk() {
        return ThreadLocalRandom.current().nextLong();
    }

    private static List<SummaryInt64Snapshot> genDataSetInt64(LongSupplier supplier) {
        List<SummaryInt64Snapshot> result = new ArrayList<>();
        SummaryInt64Snapshot prev = ImmutableSummaryInt64Snapshot.EMPTY;
        for (int index = 0; index < 500; index++) {
            long value = supplier.getAsLong();
            prev = new ImmutableSummaryInt64Snapshot(
                    prev.getCount() + 1,
                    prev.getSum() + value,
                    Math.min(prev.getMin(), value),
                    Math.max(prev.getMax(), value));
            result.add(prev);
        }

        return result;
    }

    private static List<SummaryDoubleSnapshot> genDataSetDouble(LongSupplier supplier) {
        List<SummaryDoubleSnapshot> result = new ArrayList<>();
        SummaryDoubleSnapshot prev = ImmutableSummaryDoubleSnapshot.EMPTY;
        for (int index = 0; index < 500; index++) {
            long value = supplier.getAsLong();
            prev = new ImmutableSummaryDoubleSnapshot(
                    prev.getCount() + 1,
                    prev.getSum() + value,
                    Math.min(prev.getMin(), value),
                    Math.max(prev.getMax(), value));
            result.add(prev);
        }

        return result;
    }

    private static void testGrow(String name, LongSupplier supplier) {
        testInt64("int64 -  " + name, genDataSetInt64(supplier));
        testDouble("double - " + name, genDataSetDouble(supplier));
    }

    public static void main(String[] args) {
        Random rnd = ThreadUtils.currentThreadLocalRandom();
        testGrow("rnd 0,1", () -> rnd.nextBoolean() ? 1 : 0);
        testGrow("rnd 1,2", () -> rnd.nextBoolean() ? 1 : 2);
        testGrow("rnd 2,3", () -> rnd.nextBoolean() ? 2 : 3);
        testGrow("0..1000", () -> rnd.nextInt(1000));
        testGrow("1500..1600", () -> rnd.nextInt(100) + 1500);
        testGrow("0..1000000", () -> rnd.nextInt(1000000));
        testGrow("junk", SummaryEncoderBenchmark::junk);
    }

    /*

int64 -  rnd 0,1    : 19.8 19.7 bits per point
double - rnd 0,1    : 16.6 16.5 bits per point
int64 -  rnd 1,2    : 19.8 19.7 bits per point
double - rnd 1,2    : 22.4 22.3 bits per point
int64 -  rnd 2,3    : 19.8 19.7 bits per point
double - rnd 2,3    : 22.8 22.8 bits per point
int64 -  0..1000    : 31.1 31.0 bits per point
double - 0..1000    : 34.6 34.5 bits per point
int64 -  1500..1600 : 29.4 29.3 bits per point
double - 1500..1600 : 36.2 36.2 bits per point
int64 -  0..1000000 : 39.2 39.1 bits per point
double - 0..1000000 : 41.7 41.7 bits per point
int64 -  junk       : 90.2 90.2 bits per point
double - junk       : 75.7 75.7 bits per point

     */
}
