package ru.yandex.solomon.codec.summary;

import java.util.ArrayList;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.ToLongFunction;
import java.util.stream.LongStream;
import java.util.stream.Stream;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListViewIterator;

/*

Benchmark                                             (pointCount)   Mode  Cnt      Score     Error  Units
SummaryEncoderJmhBenchmark.MultipleDouble.decompress          1000  thrpt   10   5792.533 ± 107.761  ops/s
SummaryEncoderJmhBenchmark.SummaryDouble.decompress           1000  thrpt   10  12446.059 ± 120.569  ops/s
SummaryEncoderJmhBenchmark.SummaryInt64.decompress            1000  thrpt   10  11587.846 ± 275.337  ops/s

 */

/**
 * @author Vladimir Gordiychuk
 */
public class SummaryEncoderJmhBenchmark {
    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(SummaryEncoderJmhBenchmark.class.getName())
                .detectJvmArgs()
                .build();

        new Runner(opt).run();
    }

    @Fork(value = 1)
    @Measurement(iterations = 10, time = 3, timeUnit = TimeUnit.SECONDS)
    @Warmup(iterations = 3, time = 3, timeUnit = TimeUnit.SECONDS)
    @State(Scope.Thread)
    @Threads(1) //current test not support concurrent execution
    @BenchmarkMode(Mode.Throughput)
    @OutputTimeUnit(TimeUnit.SECONDS)
    public static abstract class AbstractBenchmark {
        @Param({"1000"})
        public int pointCount;

        protected List<SummaryInt64Snapshot> generateStatistic() {
            LongSummaryStatistics latest = new LongSummaryStatistics();
            List<SummaryInt64Snapshot> statistics = new ArrayList<>();
            for (int index = 0; index < pointCount; index++) {
                LongSummaryStatistics stat = LongStream.generate(() -> ThreadLocalRandom.current().nextInt(1000))
                        .limit(10)
                        .summaryStatistics();
                latest.combine(stat);
                statistics.add(new ImmutableSummaryInt64Snapshot(latest.getCount(), latest.getSum(), latest.getMin(), latest.getMax()));
            }
            return statistics;
        }

        protected BitBuf compress(MetricType type, AggrGraphDataArrayList source) {
            TimeSeriesOutputStream stream =
                    CompressStreamFactory.createOutputStream(
                            type,
                            source.columnSetMask(),
                            source.length());

            AggrPoint point = new AggrPoint();
            AggrGraphDataArrayListViewIterator it = source.iterator();
            while (it.next(point)) {
                stream.writePoint(source.columnSetMask(), point);
            }
            return stream.getCompressedData();
        }

        protected AggrGraphDataArrayList decompress(int mask, MetricType type, BitBuf view) {
            TimeSeriesInputStream stream =
                    CompressStreamFactory.createInputStream(
                            type,
                            mask,
                            view);

            AggrPoint point = new AggrPoint();
            AggrGraphDataArrayList result = new AggrGraphDataArrayList(mask, pointCount);
            while (stream.hasNext()) {
                stream.readPoint(mask, point);
                result.addRecordData(mask, point);
            }
            return result;
        }

        @Benchmark
        public Object compress() {
            return doCompress();
        }

        @Benchmark
        public Object decompress() {
            return doDecompress();
        }

        protected abstract Object doCompress();

        protected abstract Object doDecompress();
    }

    public static class SummaryInt64 extends AbstractBenchmark {
        private AggrGraphDataArrayList source;
        private BitBuf compressed;

        @Setup()
        public void setUp() {
            int mask = StockpileColumn.TS.mask() | StockpileColumn.ISUMMARY.mask();
            source = new AggrGraphDataArrayList(mask, pointCount);
            long stepMillis = TimeUnit.SECONDS.toMillis(15L);
            long now = System.currentTimeMillis();
            AggrPointData point = new AggrPointData();
            List<SummaryInt64Snapshot> statistics = generateStatistic();
            for (SummaryInt64Snapshot stat : statistics) {
                now += stepMillis;
                point.tsMillis = now;
                point.summaryInt64 = stat;
                source.addRecordData(mask, point);
            }

            compressed = compress(MetricType.ISUMMARY, source);
        }

        @Override
        protected Object doCompress() {
            return compress(MetricType.ISUMMARY, source);
        }

        @Override
        protected Object doDecompress() {
            return decompress(source.columnSetMask(), MetricType.ISUMMARY, compressed);
        }
    }

    public static class SummaryDouble extends AbstractBenchmark {
        private AggrGraphDataArrayList source;
        private BitBuf compressed;

        @Setup()
        public void setUp() {
            int mask = StockpileColumn.TS.mask() | StockpileColumn.DSUMMARY.mask();
            source = new AggrGraphDataArrayList(mask, pointCount);
            long stepMillis = TimeUnit.SECONDS.toMillis(15L);
            long now = System.currentTimeMillis();
            AggrPointData point = new AggrPointData();
            List<SummaryInt64Snapshot> statistics = generateStatistic();
            for (SummaryInt64Snapshot stat : statistics) {
                now += stepMillis;
                point.tsMillis = now;
                point.summaryDouble = new ImmutableSummaryDoubleSnapshot(stat.getCount(), stat.getSum(), stat.getMin(), stat.getMax());
                source.addRecordData(mask, point);
            }

            compressed = compress(MetricType.DSUMMARY, source);
        }

        @Override
        protected Object doCompress() {
            return compress(MetricType.DSUMMARY, source);
        }

        @Override
        protected Object doDecompress() {
            return decompress(source.columnSetMask(), MetricType.DSUMMARY, compressed);
        }
    }

    public static class MultipleDouble extends AbstractBenchmark {
        private AggrGraphDataArrayList[] source;
        private BitBuf[] compressed;

        @Setup()
        public void setUp() {
            long now = System.currentTimeMillis();
            List<SummaryInt64Snapshot> statistics = generateStatistic();
            source = new AggrGraphDataArrayList[]{
                    prepareSource(now, statistics, SummaryInt64Snapshot::getCount),
                    prepareSource(now, statistics, SummaryInt64Snapshot::getSum),
                    prepareSource(now, statistics, SummaryInt64Snapshot::getMax),
                    prepareSource(now, statistics, SummaryInt64Snapshot::getMin)
            };
            compressed = Stream.of(source)
                    .map(list -> compress(MetricType.DGAUGE, list))
                    .toArray(BitBuf[]::new);
        }

        private AggrGraphDataArrayList prepareSource(long now, List<SummaryInt64Snapshot> statistics, ToLongFunction<SummaryInt64Snapshot> fn) {
            int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
            AggrGraphDataArrayList source = new AggrGraphDataArrayList(mask, statistics.size());
            long stepMillis = TimeUnit.SECONDS.toMillis(15L);
            AggrPointData point = new AggrPointData();
            for (SummaryInt64Snapshot stat : statistics) {
                now += stepMillis;
                point.tsMillis = now;
                point.valueDenom = ValueColumn.DEFAULT_DENOM;
                point.valueNum = fn.applyAsLong(stat);
                source.addRecordData(mask, point);
            }

            return source;
        }

        @Override
        protected Object doCompress() {
            BitBuf[] result = new BitBuf[source.length];
            for (int index = 0; index < source.length; index++) {
                result[index] = compress(MetricType.DGAUGE, source[index]);
            }
            return result;
        }

        @Override
        protected Object doDecompress() {
            AggrGraphDataArrayList[] result = new AggrGraphDataArrayList[source.length];
            for (int index = 0; index < source.length; index++) {
                result[index] = decompress(source[index].columnSetMask(), MetricType.DGAUGE, compressed[index]);
            }
            return result;
        }
    }
}
