package ru.yandex.solomon.codec.ts;

import java.time.Instant;
import java.util.Random;
import java.util.function.IntToLongFunction;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Stepan Koltsov
 */
public class TsEncoderBenchmark {

    private static void test(String name, long[] data) {
        TimeSeriesOutputStream stream = new TsOutputStream();

        AggrPoint point = new AggrPoint();
        for (long tsMillis : data) {
            point.tsMillis = tsMillis;
            stream.writePoint(StockpileColumn.TS.mask(), point);
        }

        double b = 1.0 * stream.getCompressedData().readableBits() / data.length;
        System.out.printf("%-10s: %.1f bits per point\n", name, b);
    }

    private static long[] gen(IntToLongFunction supplier) {
        return IntStream.range(0, 500).mapToLong(supplier::applyAsLong).toArray();
    }

    public static void main(String[] args) {
        Random rnd = ThreadUtils.currentThreadLocalRandom();
        long ts0 = Instant.parse("2015-11-11T22:24:54Z").toEpochMilli();
        test("consts", gen(i -> ts0));
        test("15s step", gen(i -> ts0 + i * 15000));
        test("1s step", gen(i -> ts0 + i * 1000));
        test("2h step", gen(i -> ts0 + i * 3600L * 1000));
        test("random seconds in 15s", gen(i -> ts0 + i * 15000 + rnd.nextInt(15) * 1000));
        test("random millis in 15s", gen(i -> ts0 + i * 15000 + rnd.nextInt(15000)));
    }

    @ParametersAreNonnullByDefault
    private static class TsOutputStream extends AbstractTimeSeriesOutputStream {

        public TsOutputStream() {
            super(new HeapBitBuf(), 0);
        }

        @Override
        public TimeSeriesOutputStream copy() {
            return null;
        }

        @Override
        protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {

        }

        @Override
        protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {

        }

        @Override
        protected long memorySelfSize() {
            return 0;
        }

        @Override
        protected void dumpAndResetAdditionalState(BitBuf buffer) {
        }

        @Override
        protected void restoreAdditionalState(BitBuf buffer) {
        }
    }
}
