package ru.yandex.solomon.codec.archive;

import java.util.List;

import com.google.common.base.Preconditions;

import ru.yandex.misc.algo.BinarySearch;
import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.bits.BitArray;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.frames.Frame;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.MergingAggrGraphDataIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class FramedTimeSeries {

    public static TimeSeriesOutputStream repack(TimeSeriesOutputStream out, StockpileFormat format, MetricType type, int sourceMask, int targetMask) {
        return repack(out.getCompressedData(), format, type, sourceMask, targetMask);
    }

    public static TimeSeriesOutputStream repack(BitBuf compressed, StockpileFormat format, MetricType type, int sourceMask, int targetMask) {
        StockpileColumns.ensureColumnSetValid(type, targetMask);
        var frameIt = CompressStreamFactory.createFrameIterator(compressed);
        var out = CompressStreamFactory.createOutputStream(type, targetMask, compressed.allocate(compressed.bytesSize()), 0);
        var point = RecyclableAggrPoint.newInstance();
        try {
            var frame = new Frame();
            while (frameIt.next(frame)) {
                var it = iterator(compressed, frame, format, type, sourceMask);
                while (it.next(point)) {
                    out.writePoint(targetMask, point);
                }

                if (frame.closed) {
                    out.forceCloseFrame();
                }
            }

            return out;
        } catch (Throwable e) {
            out.close();
            throw new RuntimeException(e);
        } finally {
            point.recycle();
        }
    }

    public static TimeSeriesOutputStream merge(TimeSeriesOutputStream out, StockpileFormat format, MetricType type, int mask, boolean sorted) {
        var lastFrameIdx = out.getLastFrameIdx();
        var compressed = out.getCompressedData();

        var list = sortAndMergeLastFrame(out, format, type, mask, sorted);
        if (list.isEmpty()) {
            return out.copy();
        }

        int records = 0;
        BitBuf buffer = compressed.allocate(compressed.bytesSize());
        var framesIt = CompressStreamFactory.createFrameIterator(compressed.slice(0, lastFrameIdx));

        var frame = new Frame();
        int index = 0;
        while (framesIt.next(frame)) {
            Preconditions.checkArgument(frame.closed, "Frame before last unclosed");
            if (index == list.length() || frame.lastTsMillis < list.getTsMillis(index)) {
                records += frame.records;
                buffer.writeBits(compressed.slice(frame.pos, frame.size));
                continue;
            }

            int from = index;
            index = BinarySearch.firstIndex(from, list.length(), i -> list.getTsMillis(i) > frame.lastTsMillis);

            var itFrame = iterator(compressed, frame, format, type, mask);
            var itUpdate = list.slice(from, index).iterator();
            var itMerge = MergingAggrGraphDataIterator.ofCombineAggregate(List.of(itFrame, itUpdate));

            var repacked = pack(itMerge, frame.size, format, type, mask);
            buffer.writeBits(repacked.getCompressedData());
            records += repacked.recordCount();
            repacked.close();
        }

        var result = CompressStreamFactory.createOutputStream(type, mask, buffer, records);
        var it = list.slice(index, list.length()).iterator();
        var point = RecyclableAggrPoint.newInstance();
        while (it.next(point)) {
            result.writePoint(mask, point);
        }
        point.recycle();
        list.clear();
        return result;
    }

    private static AggrGraphDataArrayList sortAndMergeLastFrame(TimeSeriesOutputStream out, StockpileFormat format, MetricType type, int mask, boolean sorted) {
        var lastFrameIdx = out.getLastFrameIdx();
        var compressed = out.getCompressedData();
        var lastFrame = compressed.slice(lastFrameIdx, compressed.writerIndex() - lastFrameIdx);

        var it = new BinaryAggrGraphDataListIterator(type, mask, lastFrame, out.frameRecordCount());
        var list = AggrGraphDataArrayList.of(it);
        if (sorted) {
            list.mergeAdjacent();
        } else {
            list.sortAndMerge();
        }
        return list;
    }

    public static AggrGraphDataListIterator iterator(BitBuf buffer, Frame frame, StockpileFormat format, MetricType type, int columnSet) {
        return new BinaryAggrGraphDataListIterator(type, columnSet, buffer.slice(frame.pos, frame.size), frame.records);
    }

    private static TimeSeriesOutputStream pack(AggrGraphDataListIterator it, long bitsCapacity, StockpileFormat format, MetricType type, int columnSet) {
        var out = CompressStreamFactory.createOutputStream(type, columnSet);
        out.ensureCapacity(BitArray.arrayLengthForBits(bitsCapacity));

        var point = RecyclableAggrPoint.newInstance();
        while (it.next(point)) {
            out.writePoint(columnSet, point);
        }
        point.recycle();
        out.forceCloseFrame();
        return out;
    }
}
