package ru.yandex.solomon.codec.archive;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.stockpile.api.EProjectId;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class MetricArchiveGeneric implements AggrGraphDataIterable, HasColumnSet, MemMeasurable, Cloneable {
    /** @see DeleteBeforeField */
    protected long deleteBefore;
    protected int ownerProjectId;
    protected int ownerShardId;

    /**
     * @see DecimPoliciesPredefined
     */
    protected int decimPolicyId;
    protected MetricType type;

    protected MetricArchiveGeneric(MetricHeader header) {
        this.deleteBefore = header.getDeleteBefore();
        this.ownerProjectId = header.getOwnerProjectId();
        this.ownerShardId = header.getOwnerShardId();
        this.decimPolicyId = header.getDecimPolicyId();
        this.type = header.getType();
    }

    protected MetricArchiveGeneric(MetricArchiveGeneric copy) {
        this.deleteBefore = copy.deleteBefore;
        this.ownerProjectId = copy.ownerProjectId;
        this.ownerShardId = copy.ownerShardId;
        this.decimPolicyId = copy.decimPolicyId;
        this.type = copy.type;
    }

    protected MetricArchiveGeneric() {
        this(MetricHeader.defaultValue);
    }

    public abstract int getRecordCount();

    @Override
    public int elapsedBytes() {
        return bytesCount();
    }

    public abstract int bytesCount();

    public boolean isEmpty() {
        return getRecordCount() == 0;
    }

    public long getDeleteBefore() {
        return deleteBefore;
    }

    public int getDecimPolicyId() {
        return decimPolicyId;
    }

    public int getOwnerProjectId() {
        return ownerProjectId;
    }

    @Nonnull
    public EProjectId getOwnerProjectIdOrUnknown() {
        return OwnerField.forNumberOrUnknown(ownerProjectId);
    }

    public int getOwnerShardId() {
        return ownerShardId;
    }

    public MetricType getType() {
        return type;
    }

    @Override
    public String toString() {
        if (isEmpty()) {
            return "{header: " + header().toString() + "}";
        }

        var point = RecyclableAggrPoint.newInstance();
        try {
            StringBuilder sb = new StringBuilder();
            sb.append("header:")
                .append(header())
                .append(", records: "+ DataSize.shortString(getRecordCount()))
                .append(", bytes: "+ DataSize.shortString(bytesCount()))
                .append(", points: [");

            if (getRecordCount() <= 10) {
                int cnt = 0;
                var it = iterator();
                while (it.next(point)) {
                    if (cnt != 0) {
                        sb.append(", ");
                    }
                    sb.append(point);
                    cnt++;
                }
            } else {
                int from = getRecordCount() - 5;
                int cnt = 0;
                var it = iterator();
                while (it.next(point)) {
                    if (cnt <= 5 || cnt >= from) {
                        if (cnt != 0) {
                            sb.append(", ");
                        }
                        if (cnt != 5) {
                            sb.append(point);
                        } else {
                            sb.append("...");
                        }
                    }
                    cnt++;
                }
            }

            sb.append("]");
            return sb.toString();
        } finally {
            point.recycle();
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        MetricArchiveGeneric that = (MetricArchiveGeneric) o;

        if (that.ownerProjectId != this.ownerProjectId) {
            return false;
        }

        if (that.ownerShardId != this.ownerShardId) {
            return false;
        }

        if (that.decimPolicyId != this.decimPolicyId) {
            return false;
        }

        if (that.deleteBefore != this.deleteBefore) {
            return false;
        }

        if (this.getRecordCount() != that.getRecordCount()) {
            return false;
        }

        var itLeft = this.iterator();
        var itRight = that.iterator();
        var pointLeft = RecyclableAggrPoint.newInstance();
        var pointRight = RecyclableAggrPoint.newInstance();

        while (itLeft.next(pointLeft) && itRight.next(pointRight)) {
            if (!pointLeft.equals(pointRight)) {
                return false;
            }
        }
        pointLeft.recycle();
        pointRight.recycle();
        return true;
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = 31 * result + (int) (deleteBefore ^ (deleteBefore >>> 32));
        result = 31 * result + ownerProjectId;
        result = 31 * result + ownerShardId;
        result = 31 * result + (int) decimPolicyId;
        result = 31 * result + getRecordCount();
        return result;
    }

    @Nonnull
    public MetricHeader header() {
        return new MetricHeader(deleteBefore, ownerProjectId, ownerShardId, decimPolicyId, type);
    }

    public abstract MetricArchiveMutable cloneToUnsealed();

    @Override
    public abstract AggrGraphDataListIterator iterator();

    @Nonnull
    public AggrGraphDataArrayList toAggrGraphDataArrayList() {
        return AggrGraphDataArrayList.of(iterator());
    }

    public long getFirstTsMillis() {
        var it = iterator();
        var point = RecyclableAggrPoint.newInstance();
        try {
            it.next(point);
            return point.tsMillis;
        } finally {
            point.recycle();
        }
    }
}
