package ru.yandex.solomon.codec.archive;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.ReadOnlyHeapBitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumnSetType;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

import static ru.yandex.solomon.model.point.column.StockpileColumns.ensureColumnSetValid;

/**
 * @author Stepan Koltsov
 * @see MetricArchiveMutable
 */
@ParametersAreNonnullByDefault
public class MetricArchiveImmutable extends MetricArchiveGeneric implements AutoCloseable {
    public static final MetricArchiveImmutable empty = new MetricArchiveImmutable(
            MetricHeader.defaultValue,
            StockpileFormat.CURRENT,
            StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask(),
            ReadOnlyHeapBitBuf.EMPTY,
            0);

    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MetricArchiveImmutable.class);

    @StockpileColumnSetType
    private final int columnSetMask;
    private final BitBuf buffer;
    private final int recordCount;
    private final StockpileFormat format;

    public MetricArchiveImmutable(
            MetricHeader header,
            StockpileFormat format,
            int columnSetMask,
            BitBuf buffer,
            int recordCount)
    {
        super(header);
        this.format = format;
        this.buffer = buffer;
        this.recordCount = recordCount;
        if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
            this.type = StockpileColumns.typeByMask(columnSetMask);
        }

        if (columnSetMask == StockpileColumnSet.empty.columnSetMask()) {
            this.columnSetMask = StockpileColumnSet.empty.columnSetMask();
        } else {
            ensureColumnSetValid(type, columnSetMask);
            this.columnSetMask = columnSetMask;
        }
    }

    public static MetricArchiveImmutable of(AggrGraphDataIterable source) {
        return of(StockpileColumns.typeByMask(source.columnSetMask()), source);
    }

    public static MetricArchiveImmutable of(MetricType type, AggrGraphDataIterable source) {
        try (var mutable = MetricArchiveMutable.of(type, source)) {
            return mutable.toImmutableNoCopy();
        }
    }

    public static MetricArchiveImmutable of(AggrGraphDataListIterator it) {
        try (var mutable = MetricArchiveMutable.of(it)) {
            return mutable.toImmutableNoCopy();
        }
    }

    public static MetricArchiveImmutable of(AggrPoint... points) {
        try (var mutable = MetricArchiveMutable.of(AggrGraphDataArrayList.of(points))) {
            return mutable.toImmutableNoCopy();
        }
    }

    @Override
    public int getRecordCount() {
        return recordCount;
    }

    @Override
    public int bytesCount() {
        return buffer.bytesSize();
    }

    @Override
    public int columnSetMask() {
        return columnSetMask;
    }

    public StockpileFormat getFormat() {
        return format;
    }

    @Override
    protected MetricArchiveImmutable clone() {
        try {
            return (MetricArchiveImmutable) super.clone();
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    public MetricArchiveMutable cloneToUnsealed() {
        MetricArchiveMutable result = new MetricArchiveMutable(header());
        // TODO: restore from Immutable to Mutable archive CPU expensive operation,
        // for avoid it maybe need write for immutable archive the latest few bytes as
        // latest state for encoder(previous values) in that case we avoid repack whole archive
        result.addAll(this);
        return result;
    }

    public MetricArchiveMutable toMutable() {
        return new MetricArchiveMutable(header(), format, columnSetMask, buffer.duplicate());
    }

    @Override
    public AggrGraphDataListIterator iterator() {
        return new BinaryAggrGraphDataListIterator(type, columnSetMask, buffer.duplicate(), recordCount);
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + buffer.memorySizeIncludingSelf();
    }

    public BitBuf getCompressedDataRaw() {
        return buffer;
    }

    @Override
    public void close() {
        buffer.release();
    }
}
