package ru.yandex.solomon.codec.archive;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.PublicCloneable;
import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.archive.header.DecimPolicyField;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.archive.header.MetricTypeField;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.BitBufAllocator;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumnSetType;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.predicate.AggrPointPredicate;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.AggrGraphDataLists;
import ru.yandex.solomon.model.timeseries.AggrGraphDataSink;
import ru.yandex.solomon.model.timeseries.FilteringAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.FilteringBeforeAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.stockpile.api.EProjectId;

/**
 * @author Stepan Koltsov
 *
 * @see MetricArchiveImmutable
 */
@ParametersAreNonnullByDefault
public class MetricArchiveMutable extends MetricArchiveGeneric implements AggrGraphDataSink, PublicCloneable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MetricArchiveMutable.class);

    private final StockpileFormat format;
    @StockpileColumnSetType
    private int columnSetMask;
    private TimeSeriesOutputStream dataStream = CompressStreamFactory.emptyOutputStream();

    private long lastTsMillis;
    private int outOfOrderCount;
    private boolean containDuplicates;

    public MetricArchiveMutable() {
        this(MetricHeader.defaultValue);
    }

    public MetricArchiveMutable(MetricHeader header) {
        super(header);
        this.format = StockpileFormat.CURRENT;
    }

    public MetricArchiveMutable(MetricHeader header, StockpileFormat format) {
        super(header);
        this.format = format;
    }

    public MetricArchiveMutable(MetricHeader header, StockpileFormat format, int columnSetMask, BitBuf buffer) {
        super(header);
        this.format = format;
        this.columnSetMask = columnSetMask;
        this.dataStream = CompressStreamFactory.restoreOutputStream(header.getType(), columnSetMask, buffer);
        this.lastTsMillis = dataStream.getLastTsMillis();
    }

    public MetricArchiveMutable(MetricArchiveMutable copy) {
        super(copy);
        this.columnSetMask = copy.columnSetMask;
        this.lastTsMillis = copy.lastTsMillis;
        this.outOfOrderCount = copy.outOfOrderCount;
        this.containDuplicates = copy.containDuplicates;
        this.dataStream = copy.dataStream.copy();
        this.format = copy.format;
    }

    public static MetricArchiveMutable of(AggrGraphDataIterable source) {
        return of(source.iterator());
    }

    public static MetricArchiveMutable of(MetricType type, AggrGraphDataIterable source) {
        return of(type, source.iterator());
    }

    public static MetricArchiveMutable of(AggrGraphDataListIterator it) {
        return of(StockpileColumns.typeByMask(it.columnSet), it);
    }

    public static MetricArchiveMutable of(MetricType type, AggrGraphDataListIterator it) {
        MetricArchiveMutable content = new MetricArchiveMutable();
        content.setType(type);
        content.addAllFrom(it);
        return content;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return getSelfSize() + dataStream.memorySizeIncludingSelfInt();
    }

    protected long getSelfSize() {
        return SELF_SIZE;
    }

    @Override
    public int getRecordCount() {
        return dataStream.recordCount();
    }

    @Override
    public int bytesCount() {
        return dataStream.bytesCount();
    }

    @Override
    public int columnSetMask() {
        if (columnSetMask == 0) {
            return StockpileColumns.minColumnSet(type);
        }
        return columnSetMask;
    }

    @Override
    public AggrGraphDataListIterator iterator() {
        return iterator(dataStream.getCompressedData(), dataStream.recordCount());
    }

    private AggrGraphDataListIterator iterator(BitBuf compressed, int records) {
        return new BinaryAggrGraphDataListIterator(type, columnSetMask, compressed, records);
    }

    @Override
    public void ensureCapacity(int columnSetMask, int capacity) {
        if (this.columnSetMask == 0) {
            if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
                type = StockpileColumns.typeByMask(columnSetMask);
            }
            this.columnSetMask = StockpileColumns.minColumnSet(type) | columnSetMask;
            if (this.columnSetMask != 0) {
                this.dataStream = CompressStreamFactory.createOutputStream(type, this.columnSetMask, allocateBuffer(capacity, bytesCount()), 0);
            }
            return;
        }

        expandTo(columnSetMask);
        dataStream.ensureCapacity(columnSetMask, capacity);
    }

    public void ensureCapacity(int bytes) {
        dataStream.ensureCapacity(bytes);
    }

    public void ensureBytesCapacity(int columnSetMask, int bytes) {
        expandTo(columnSetMask);
        dataStream.ensureCapacity(bytes);
    }

    @Override
    public void addRecordData(int columnSetMask, AggrPointData data) {
        expandTo(columnSetMask);
        writePoint(data);
        checkTooManyOutOfOrder();
    }

    @Override
    public void addAllFrom(AggrGraphDataListIterator iterator) {
        addAllNoSortMergeFrom(iterator);
        checkTooManyOutOfOrder();
    }

    @Override
    public MetricArchiveMutable clone() {
        return cloneToUnsealed();
    }

    @Override
    public MetricArchiveMutable cloneToUnsealed() {
        return new MetricArchiveMutable(this);
    }

    /**
     * Return sorted and merged list
     */
    @Nonnull
    @Override
    public AggrGraphDataArrayList toAggrGraphDataArrayList() {
        AggrGraphDataArrayList list = AggrGraphDataLists.toAggrListUnsorted(this);
        if (isSorted()) {
            if (containDuplicates) {
                list.mergeAdjacent();
            }
            return list;
        }

        list.sortAndMerge();
        return list;
    }

    public StockpileFormat getFormat() {
        return format;
    }

    public void setDeleteBefore(long deleteBefore) {
        this.deleteBefore = DeleteBeforeField.merge(this.deleteBefore, deleteBefore);
        if (deleteBefore == DeleteBeforeField.DELETE_ALL) {
            clear();
        } else if (deleteBefore == DeleteBeforeField.KEEP || isEmpty()) {
            return;
        }

        var compressed = dataStream.getCompressedData().retain();
        try {
            int records = dataStream.recordCount();
            var it = FilteringBeforeAggrGraphDataIterator.of(deleteBefore, iterator(compressed, records));
            clear();
            addAllFrom(it);
        } finally {
            compressed.release();
        }
    }

    public void setOwnerProjectId(int ownerProjectId) {
        this.ownerProjectId = OwnerField.mergeOwnerField(this.ownerProjectId, ownerProjectId);
    }

    public void setOwnerProjectIdEnum(EProjectId projectId) {
        setOwnerProjectId(projectId.getNumber());
    }

    public void setOwnerShardId(int ownerShardId) {
        this.ownerShardId = OwnerField.mergeOwnerField(this.ownerShardId, ownerShardId);
    }

    public void setType(MetricType type) {
        MetricType merged = MetricTypeField.merge(this.type, type);
        if (this.type == merged || merged == MetricType.METRIC_TYPE_UNSPECIFIED) {
            return;
        }

        if (columnSetMask == 0 || getRecordCount() == 0) {
            this.type = merged;
            this.columnSetMask = StockpileColumns.minColumnSet(merged);
            var buffer = allocateBuffer(getRecordCount(), bytesCount());
            this.dataStream.close();
            this.dataStream = CompressStreamFactory.createOutputStream(merged, columnSetMask, buffer, 0);
            return;
        }

        sortAndMerge();
        AggrGraphDataListIterator it = MetricTypeTransfers.of(this.type, merged, iterator());
        int mask = it.columnSetMask();
        var buffer = allocateBuffer(getRecordCount(), bytesCount());
        TimeSeriesOutputStream repackStream = CompressStreamFactory.createOutputStream(merged, mask, buffer, 0);

        var point = RecyclableAggrPoint.newInstance();
        while (it.next(point)) {
            repackStream.writePoint(mask, point);
        }
        point.recycle();

        this.type = merged;
        this.columnSetMask = mask;
        this.dataStream.close();
        this.dataStream = repackStream;
    }

    public void setDecimPolicyId(int decimPolicyId) {
        this.decimPolicyId = DecimPolicyField.merge(this.decimPolicyId, decimPolicyId);
    }

    public void updateWith(MetricArchiveGeneric metricArchiveGeneric) {
        updateWith(metricArchiveGeneric, AggrPointPredicate.TRUE);
    }

    public void updateWith(MetricArchiveGeneric archive, AggrPointPredicate pointPredicate) {
        setDeleteBefore(archive.deleteBefore);
        setOwnerProjectId(archive.ownerProjectId);
        setOwnerShardId(archive.ownerShardId);
        setDecimPolicyId(archive.decimPolicyId);
        setType(archive.getType());
        ensureBytesCapacity(archive.columnSetMask(), archive.bytesCount());
        addAllFrom(FilteringAggrGraphDataIterator.of(archive.iterator(), pointPredicate));
    }

    public void updateWithNoSortMerge(MetricArchiveGeneric archive) {
        setDeleteBefore(archive.deleteBefore);
        setOwnerProjectId(archive.ownerProjectId);
        setOwnerShardId(archive.ownerShardId);
        setDecimPolicyId(archive.decimPolicyId);
        setType(archive.getType());
        addAllNoSortMergeFrom(archive.iterator());
    }

    public MetricArchiveImmutable toImmutable() {
        if (isSorted() && !containDuplicates) {
            return toImmutableNoCopy();
        }

        try (var mutable = new MetricArchiveMutable(this)) {
            return mutable.toImmutableNoCopy();
        }
    }

    public MetricArchiveImmutable toImmutableNoCopy() {
        sortAndMerge();
        return new MetricArchiveImmutable(header(), format,
                columnSetMask,
                dataStream.getCompressedData().retain(),
                dataStream.recordCount());
    }

    public long getLastTsMillis() {
        return lastTsMillis;
    }

    public void sortAndMerge() {
        if (isSortedAndMerged()) {
            return;
        }

        var merged = FramedTimeSeries.merge(dataStream, format, type, columnSetMask, isSorted());
        this.outOfOrderCount = 0;
        this.containDuplicates = false;
        this.dataStream.close();
        this.dataStream = merged;
    }

    public BitBuf getCompressedDataRaw() {
        return dataStream.getCompressedData();
    }

    public void addAllNoSortMergeFrom(AggrGraphDataListIterator iterator) {
        ensureCapacity(iterator.columnSetMask(), iterator.estimatePointsCount());
        var point = RecyclableAggrPoint.newInstance();
        try {
            while (iterator.next(point)) {
                writePoint(point);
            }
        } finally {
            point.recycle();
        }
    }

    private void expandTo(int targetColumnSetMask) {
        if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
            type = StockpileColumns.typeByMask(targetColumnSetMask);
        }

        if (this.columnSetMask == 0) {
            columnSetMask = StockpileColumns.minColumnSet(type) | targetColumnSetMask;
            if (columnSetMask != 0) {
                dataStream.close();
                dataStream = CompressStreamFactory.createOutputStream(type, columnSetMask, allocateBuffer(getRecordCount(), bytesCount()), 0);
            }
            return;
        }

        if (!StockpileColumnSet.needToAddAtLeastOneColumn(columnSetMask, targetColumnSetMask)) {
            return;
        }

        int resultColumnSet = this.columnSetMask | targetColumnSetMask;
        var out = FramedTimeSeries.repack(dataStream, format, type, columnSetMask, resultColumnSet);
        this.columnSetMask = resultColumnSet;
        this.dataStream.close();
        this.dataStream = out;
    }

    private void writePoint(AggrPointData pointData) {
        dataStream.writePoint(this.columnSetMask, pointData);
        if (pointData.tsMillis < lastTsMillis || outOfOrderCount > 0) {
            ++outOfOrderCount;
        } else if (pointData.tsMillis == lastTsMillis) {
            containDuplicates = true;
        }
        lastTsMillis = Math.max(lastTsMillis, pointData.tsMillis);
    }

    private void checkTooManyOutOfOrder() {
        if (outOfOrderCount > 20) {
            if (outOfOrderCount > dataStream.recordCount() / 2) {
                sortAndMerge();
            }
        }
    }

    protected boolean isSorted() {
        return outOfOrderCount == 0;
    }

    protected boolean hasDuplicates() {
        return containDuplicates;
    }

    public boolean isSortedAndMerged() {
        return isSorted() && !containDuplicates;
    }

    /**
     * @return last frame bytes count
     */
    public int frameBytesCount() {
        return dataStream.frameBytesCount();
    }

    /**
     * @return last frame records count
     */
    public int frameRecordCount() {
        return dataStream.frameRecordCount();
    }

    /**
     * close last frame if enough data for it
     */
    public boolean closeFrame() {
        sortAndMerge();
        return dataStream.closeFrame();
    }

    public void forceCloseFrame() {
        sortAndMerge();
        dataStream.forceCloseFrame();
    }

    private void clear() {
        this.columnSetMask = 0;
        this.outOfOrderCount = 0;
        this.containDuplicates = false;
        this.lastTsMillis = 0;
        this.dataStream.close();
        this.dataStream = CompressStreamFactory.emptyOutputStream();
    }

    protected long getLatestFramePos() {
        return dataStream.getLastFrameIdx();
    }

    protected BitBuf allocateBuffer(int records, int bytes) {
        return BitBufAllocator.buffer(Math.max(16, bytes));
    }

    @Override
    public void close() {
        dataStream.close();
        dataStream = CompressStreamFactory.emptyOutputStream();
    }
}
