package ru.yandex.solomon.codec.archive;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nullable;

import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.FilteringBeforeAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.MergingAggrGraphDataIterator;

/**
 * @author Sergey Polovko
 */
public final class MetricArchiveUtils {
    private MetricArchiveUtils() {}


    public static MetricArchiveMutable merge(MetricArchiveImmutable... archives) {
        return merge(Arrays.asList(archives));
    }

    public static MetricArchiveMutable merge(Iterable<MetricArchiveImmutable> archives) {
        MetricArchiveMutable result = new MetricArchiveMutable();
        for (MetricArchiveImmutable input : archives) {
            result.updateWith(input);
        }
        return result;
    }

    public static AggrGraphDataArrayList mergeToList(MetricArchiveImmutable... archives) {
        return mergeToList(Arrays.asList(archives));
    }

    public static AggrGraphDataArrayList mergeToList(List<MetricArchiveImmutable> archives) {
        List<AggrGraphDataListIterator> iterators = new ArrayList<>(archives.size());
        int mask = StockpileColumnSet.empty.columnSetMask();
        int capacity = 0;
        // TODO: backward compatibility, but it's not correct i think because merge process use max over all delete before (gordiychuk@)
        long deleteBefore = DeleteBeforeField.KEEP;
        for (MetricArchiveImmutable archive : archives) {
            mask |= archive.columnSetMask();
            capacity += archive.getRecordCount();
            deleteBefore = archive.getDeleteBefore();
            if (archive.getRecordCount() > 0) {
                iterators.add(archive.iterator());
            }
        }

        AggrGraphDataArrayList list = new AggrGraphDataArrayList(mask, capacity);
        list.addAllFrom(FilteringBeforeAggrGraphDataIterator.of(
                deleteBefore,
                MergingAggrGraphDataIterator.ofCombineAggregate(iterators)));
        return list;
    }

    public static void validate(MetricArchiveImmutable a) {
        AggrGraphDataListIterator it = a.iterator();

        // should not fail
        AggrPoint devNull = new AggrPoint();
        do {} while (it.next(devNull));
    }

    public static MetricArchiveImmutable repack(StockpileFormat format, MetricArchiveImmutable archive) {
        if (archive.getFormat() == format) {
            return archive;
        }

        try (var result = new MetricArchiveMutable(archive.header(), format)) {
            result.ensureBytesCapacity(archive.columnSetMask(), archive.bytesCount());
            result.addAllFrom(archive.iterator());
            return result.toImmutableNoCopy();
        }
    }

    public static MetricArchiveMutable repack(StockpileFormat format, MetricArchiveMutable archive) {
        if (archive.getFormat() == format) {
            return archive;
        }

        MetricArchiveMutable result = new MetricArchiveMutable(archive.header(), format);
        result.ensureCapacity(archive.columnSetMask(), archive.getRecordCount());
        result.ensureBytesCapacity(archive.columnSetMask(), archive.bytesCount());
        result.addAllFrom(archive.iterator());
        return result;
    }

    public static MetricArchiveMutable copy(StockpileFormat format, MetricArchiveMutable archive) {
        if (archive.getFormat() == format) {
            return new MetricArchiveMutable(archive);
        }

        MetricArchiveMutable result = new MetricArchiveMutable(archive.header(), format);
        result.ensureCapacity(archive.columnSetMask(), archive.getRecordCount());
        result.addAllFrom(archive.iterator());
        return result;
    }

    public static MetricArchiveImmutable encode(StockpileFormat format, AggrGraphDataIterable source) {
        if (source instanceof MetricArchiveImmutable) {
            return repack(format, (MetricArchiveImmutable) source);
        }

        try (var result = new MetricArchiveMutable(MetricHeader.defaultValue, format)) {
            result.addAll(source);
            return result.toImmutableNoCopy();
        }
    }

    public static MetricArchiveImmutable encode(StockpileFormat format, Metric<MetricId> metric) {
        var source = metric.getTimeseries();
        if (source instanceof MetricArchiveImmutable) {
            return repack(format, (MetricArchiveImmutable) source);
        }

        var header = MetricHeader.defaultValue.withKind(metric.getType());
        try (var archive = new MetricArchiveMutable(header, format)) {
            archive.ensureBytesCapacity(source.columnSetMask(), source.elapsedBytes());
            archive.addAllFrom(source.iterator());
            return archive.toImmutableNoCopy();
        }
    }

    @Nullable
    public static TimeSeries.Chunk makeChunk(StockpileFormat format, Metric<MetricId> metric, long fromMillis, long toMillis) {
        if (metric.getTimeseries() == null || metric.getTimeseries().isEmpty()) {
            return null;
        }

        try (var archive = encode(format, metric)) {
            if (archive.isEmpty()) {
                return null;
            }

            var serializer = MetricArchiveNakedSerializer.serializerForFormatSealed(format);
            return TimeSeries.Chunk.newBuilder()
                    .setFromMillis(fromMillis)
                    .setToMillis(toMillis)
                    .setPointCount(archive.getRecordCount())
                    .setContent(serializer.serializeToByteString(archive))
                    .build();
        }
    }
}
