package ru.yandex.solomon.codec.archive.header;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;

/**
 * @author Maksim Leonov
 */
@ParametersAreNonnullByDefault
public class MetricHeader {
    public static final MetricHeader defaultValue =
        new MetricHeader(DeleteBeforeField.KEEP, 0, 0, (short) 0, MetricType.METRIC_TYPE_UNSPECIFIED);

    /** @see DeleteBeforeField */
    private final long deleteBefore;
    private final int ownerProjectId;
    private final int ownerShardId;
    private final int decimPolicyId;
    private final MetricType type;

    public MetricHeader(long deleteBefore, int ownerProjectId, int ownerShardId, int decimPolicyId, MetricType type) {
        this.deleteBefore = deleteBefore;
        this.ownerProjectId = ownerProjectId;
        this.ownerShardId = ownerShardId;
        this.decimPolicyId = decimPolicyId;
        this.type = type;

        if (ownerProjectId < 0) {
            throw new IllegalArgumentException("negative ownerProjectId");
        }
        if (decimPolicyId < 0) {
            throw new IllegalArgumentException("negative decimPolicyId");
        }
    }

    public long getDeleteBefore() {
        return deleteBefore;
    }

    public int getOwnerProjectId() {
        return ownerProjectId;
    }

    public EProjectId getOwnerProject() {
        return EProjectId.forNumber(ownerProjectId);
    }

    public int getOwnerShardId() {
        return ownerShardId;
    }

    public int getDecimPolicyId() {
        return decimPolicyId;
    }

    public MetricType getType() {
        return type;
    }

    public static MetricHeader merge(MetricHeader a, MetricHeader b) {
        return new MetricHeader(
            DeleteBeforeField.merge(a.deleteBefore, b.deleteBefore),
            OwnerField.mergeOwnerField(a.ownerProjectId, b.ownerProjectId),
            OwnerField.mergeOwnerField(a.ownerShardId, b.ownerShardId),
            DecimPolicyField.merge(a.decimPolicyId, b.decimPolicyId),
            MetricTypeField.merge(a.type, b.type)
        );
    }

    public MetricHeader withKind(MetricType type) {
        return new MetricHeader(deleteBefore, ownerProjectId, ownerShardId, decimPolicyId, type);
    }

    public MetricHeader withDecimPolicy(short decimPolicyId) {
        return new MetricHeader(deleteBefore, ownerProjectId, ownerShardId, decimPolicyId, type);
    }

    public MetricHeader withDeleteBefore(long deleteBefore) {
        return new MetricHeader(deleteBefore, ownerProjectId, ownerShardId, decimPolicyId, type);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        MetricHeader that = (MetricHeader) o;

        if (deleteBefore != that.deleteBefore) return false;
        if (ownerProjectId != that.ownerProjectId) return false;
        if (ownerShardId != that.ownerShardId) return false;
        if (decimPolicyId != that.decimPolicyId) return false;
        return type == that.type;
    }

    @Override
    public int hashCode() {
        int result = (int) (deleteBefore ^ (deleteBefore >>> 32));
        result = 31 * result + ownerProjectId;
        result = 31 * result + ownerShardId;
        result = 31 * result + (int) decimPolicyId;
        result = 31 * result + (type != null ? type.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "MetricHeader{" +
            "deleteBefore=" + deleteBefore +
            ", ownerProjectId=" + EProjectId.forNumber(ownerProjectId) +
            ", ownerShardId=" + Integer.toUnsignedLong(ownerShardId) +
            ", decimPolicyId=" + EDecimPolicy.forNumber(decimPolicyId) +
            ", type=" + type +
            '}';
    }
}
