package ru.yandex.solomon.codec.archive.serializer;

import java.util.EnumMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.ReadOnlyHeapBitBuf;
import ru.yandex.solomon.codec.serializer.SelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.array.BitBufSerializer;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MetricArchiveImmutableSelfContainedSerializer {
    private static final EnumMap<StockpileFormat, SelfContainedSerializer<MetricArchiveImmutable>> byFormat =
            StockpileFormat.makeMap(ImplV2::new);

    public static SelfContainedSerializer<MetricArchiveImmutable> makeSerializerForVersion(StockpileFormat format) {
        return byFormat.get(format);
    }

    private static class ImplV2 implements SelfContainedSerializer<MetricArchiveImmutable> {
        private final StockpileFormat format;

        public ImplV2(StockpileFormat format) {
            this.format = format;
        }

        @Override
        public void serializeWithLength(MetricArchiveImmutable archive, StockpileSerializer serializer) {
            var source = archive;
            archive = MetricArchiveUtils.repack(format, source);
            MetricHeaderSerializer.I.serializeWithLength(archive.header(), serializer);

            if (archive.isEmpty()) {
                // TODO: remove or serialize more useful information
                serializer.writeVarint32(0); // archive count
                return;
            }

            // TODO: remove or serialize more useful information
            serializer.writeVarint32(1); // archive count
            // TODO: remove or serialize more useful information
            serializer.writeVarint32(0); // step millis (not used anymore)

            serializer.writeVarint32(archive.getRecordCount());
            StockpileColumnSetSerializer.writeColumnSet(serializer, archive.columnSetMask());
            BitBuf compressedData = archive.getCompressedDataRaw();
            BitBufSerializer.I.serializeWithLength(compressedData, serializer);
            if (source != archive) {
                archive.close();
            }
        }

        @Override
        public MetricArchiveImmutable deserializeWithLength(StockpileDeserializer deserializer) {
            MetricHeader header = MetricHeaderSerializer.I.deserializeWithLength(deserializer);

            // TODO: remove or use for useful information
            int archiveCount = deserializer.readVarint32();
            if (archiveCount == 0) {
                return new MetricArchiveImmutable(header, format, 0, ReadOnlyHeapBitBuf.EMPTY, 0);
            }

            if (archiveCount != 1) {
                throw new AssertionError("there must be no or only one archive, got: " + archiveCount);
            }

            // TODO: remove or use for useful information
            int stepMillis = deserializer.readVarint32();
            if (stepMillis != 0) {
                throw new AssertionError("stepMillis must be equal to 0, got: " + stepMillis);
            }

            int recordCount = deserializer.readVarint32();
            int columnSetMask = StockpileColumnSetSerializer.readColumnSet(deserializer);
            BitBuf compressedData = BitBufSerializer.I.deserializeWithLength(deserializer);
            return new MetricArchiveImmutable(header, format, columnSetMask, compressedData.asReadOnly(), recordCount);
        }
    }
}
