package ru.yandex.solomon.codec.archive.serializer;

import java.util.EnumMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.serializer.SelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.array.BitBufSerializer;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MetricArchiveMutableSelfContainedSerializer {
    private static final EnumMap<StockpileFormat, SelfContainedSerializer<MetricArchiveMutable>> byFormat =
            StockpileFormat.makeMap(ImplV2::new);

    public static SelfContainedSerializer<MetricArchiveMutable> makeSerializerForVersion(StockpileFormat format) {
        return byFormat.get(format);
    }

    private static class ImplV2 implements SelfContainedSerializer<MetricArchiveMutable> {

        private final StockpileFormat format;

        public ImplV2(StockpileFormat format) {
            this.format = format;
        }

        @Override
        public void serializeWithLength(MetricArchiveMutable archive, StockpileSerializer serializer) {
            var repacked = MetricArchiveUtils.repack(format, archive);
            try {
                MetricHeaderSerializer.I.serializeWithLength(repacked.header(), serializer);

                if (archive.isEmpty()) {
                    // TODO: remove or serialize more useful information
                    serializer.writeVarint32(0);
                    return;
                }

                // TODO: remove or serialize more useful information
                serializer.writeVarint32(1); // archive count
                // TODO: remove or serialize more useful information
                serializer.writeVarint32(0); // step millis (for backward compatibility)

                repacked.sortAndMerge();
                serializer.writeVarint32(repacked.getRecordCount());
                StockpileColumnSetSerializer.writeColumnSet(serializer, repacked.columnSetMask());
                BitBufSerializer.I.serializeWithLength(repacked.getCompressedDataRaw(), serializer);
            } finally {
                if (repacked != archive) {
                    repacked.close();
                }
            }
        }

        @Override
        public MetricArchiveMutable deserializeWithLength(StockpileDeserializer deserializer) {
            MetricHeader header = MetricHeaderSerializer.I.deserializeWithLength(deserializer);

            MetricArchiveMutable result = new MetricArchiveMutable(header);
            // TODO: remove or use for useful information
            int archiveCount = deserializer.readVarint32();
            if (archiveCount == 0) {
                return result;
            }

            if (archiveCount != 1) {
                throw new AssertionError("there must be no or only one archive, got: " + archiveCount);
            }

            // TODO: remove or use for useful information
            int stepMillis = deserializer.readVarint32();
            if (stepMillis != 0) {
                throw new AssertionError("stepMillis must be equal to 0, got: " + stepMillis);
            }

            int recordCount = deserializer.readVarint32();
            int columnSetMask = StockpileColumnSetSerializer.readColumnSet(deserializer);
            BitBuf compressedData = BitBufSerializer.I.deserializeWithLength(deserializer);
            try {
                MetricType type = header.getType() != MetricType.METRIC_TYPE_UNSPECIFIED
                    ? header.getType()
                    : StockpileColumns.typeByMask(columnSetMask);
                result.ensureBytesCapacity(columnSetMask, compressedData.bytesSize());
                result.addAllFrom(new BinaryAggrGraphDataListIterator(type, columnSetMask, compressedData, recordCount));
                return result;
            } finally {
                compressedData.release();
            }
        }
    }
}
