package ru.yandex.solomon.codec.archive.serializer;

import java.util.EnumMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.ReadOnlyHeapBitBuf;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.array.BitBufSerializer;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializer;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class MetricArchiveNakedSerializer {
    private static final EnumMap<StockpileFormat, NakedSerializer<MetricArchiveImmutable>> byVersionSealed =
        StockpileFormat.makeMap(ImplV2::new);

    public static NakedSerializer<MetricArchiveImmutable> serializerForFormatSealed(StockpileFormat format) {
        return byVersionSealed.get(format);
    }

    private static class ImplV2 implements NakedSerializer<MetricArchiveImmutable> {
        private final StockpileFormat format;

        public ImplV2(StockpileFormat format) {
            this.format = format;
        }

        @Override
        public MetricArchiveImmutable deserializeToEofImpl(StockpileDeserializer deserializer) {
            MetricHeader header = MetricHeaderSerializer.I.deserializeWithLength(deserializer);
            if (!deserializer.readBoolean()) {
                return new MetricArchiveImmutable(header, format, 0, ReadOnlyHeapBitBuf.EMPTY, 0);
            }

            int recordCount = deserializer.readVarint32();
            int columnSetMask = StockpileColumnSetSerializer.readColumnSet(deserializer);
            BitBuf compressedData = BitBufSerializer.I.deserializeWithLength(deserializer);
            return new MetricArchiveImmutable(header, format, columnSetMask, compressedData.asReadOnly(), recordCount);
        }

        @Override
        public void serializeToEof(MetricArchiveImmutable archive, StockpileSerializer serializer) {
            var source = archive;
            archive = MetricArchiveUtils.repack(format, source);
            MetricHeaderSerializer.I.serializeWithLength(archive.header(), serializer);
            if (archive.isEmpty()) {
                serializer.writeBoolean(false);
            } else {
                serializer.writeBoolean(true);
                serializer.writeVarint32(archive.getRecordCount());
                StockpileColumnSetSerializer.writeColumnSet(serializer, archive.columnSetMask());
                BitBuf compressedData = archive.getCompressedDataRaw();
                BitBufSerializer.I.serializeWithLength(compressedData, serializer);
            }
            if (archive != source) {
                archive.close();
            }
        }
    }
}
