package ru.yandex.solomon.codec.bits;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import io.netty.util.ReferenceCounted;

import ru.yandex.solomon.memory.layout.MemMeasurable;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class BitBuf implements MemMeasurable, ReferenceCounted {
    public abstract void ensureBytesCapacity(int minBytesCapacity);

    public abstract void writeBit(boolean bit);

    public void write8Bits(int bits) {
        write8Bits((byte) bits);
    }

    public abstract void write8Bits(byte bits);

    public abstract void write32Bits(int bits);

    public abstract void write64Bits(long bits);

    public void writeBits(byte bits, int count) {
        if (count > Byte.SIZE) {
            throw new IllegalArgumentException("cannot write more than " + Byte.SIZE + " bits: " + count);
        }
        writeBits(Byte.toUnsignedLong(bits), count);
    }

    public abstract void writeBits(int bits, int count);

    public abstract void writeBits(long bits, int count);

    public void writeFloatBits(float f) {
        write32Bits(Float.floatToRawIntBits(f));
    }

    public void writeDoubleBits(double d) {
        write64Bits(Double.doubleToRawLongBits(d));
    }

    public abstract void writeIntVarint1N(int n, int max);

    /**
     * Compatible with protobuf.
     *
     * @see CodedOutputStream#writeRawVarint32(int)
     */
    public void writeIntVarint8(int n) {
        for (;;) {
            if ((n & ~0x7f) == 0) {
                write8Bits(n);
                return;
            }
            write8Bits(((n & 0x7f) | 0x80));
            n >>>= 7;
        }
    }

    public void writeIntZigzagVarint8(int n) {
        writeIntVarint8(CodedOutputStream.encodeZigZag32(n));
    }

    /**
     * Compatible with protobuf.
     *
     * @see CodedOutputStream#writeRawVarint64(long)
     */
    public void writeLongVarint8(long n) {
        for (;;) {
            if ((n & ~0x7f) == 0) {
                write8Bits((int) n);
                return;
            }
            write8Bits((int) ((n & 0x7f) | 0x80));
            n >>>= 7;
        }
    }

    public void writeLongZigzagVarint8(long n) {
        writeLongVarint8(CodedOutputStream.encodeZigZag64(n));
    }

    /**
     * Transfers the specified source buffer's data to this buffer starting at
     * the current {@code writerIndex} until the source buffer becomes
     * unreadable, and increases the {@code writerIndex} by the number of
     * the transferred bytes.  This method is basically same with
     * {@link #writeBits(BitBuf, long, long)}, except that this method
     * increases the {@code readerIndex} of the source buffer by the number of
     * the transferred bytes while {@link #writeBits(BitBuf, long, long)}
     * does not.
     *
     * @throws IndexOutOfBoundsException
     *         if {@code src.readableBytes} is greater than
     *            {@code this.writableBytes}
     */
    public void writeBits(BitBuf src) {
        writeBits(src, src.readableBits());
    }

    /**
     * Transfers the specified source buffer's data to this buffer starting at
     * the current {@code writerIndex} and increases the {@code writerIndex}
     * by the number of the transferred bytes (= {@code length}).  This method
     * is basically same with {@link #writeBits(BitBuf, long, long)},
     * except that this method increases the {@code readerIndex} of the source
     * buffer by the number of the transferred bytes (= {@code length}) while
     * {@link #writeBits(BitBuf, long, long)} does not.
     *
     * @param length the number of bytes to transfer
     *
     * @throws IndexOutOfBoundsException
     *         if {@code length} is greater than {@code this.writableBytes} or
     *         if {@code length} is greater then {@code src.readableBytes}
     */
    public void writeBits(BitBuf src, long length) {
        var readerIdx = src.readerIndex();
        writeBits(src, readerIdx, length);
        src.readerIndex(readerIdx + length);
    }

    /**
     * Transfers the specified source buffer's data to this buffer starting at
     * the current {@code writerIndex} and increases the {@code writerIndex}
     * by the number of the transferred bytes (= {@code length}).
     *
     * @param srcIndex the first index of the source
     * @param length   the number of bytes to transfer
     *
     * @throws IndexOutOfBoundsException
     *         if the specified {@code srcIndex} is less than {@code 0},
     *         if {@code srcIndex + length} is greater than
     *            {@code src.capacity}, or
     *         if {@code length} is greater than {@code this.writableBytes}
     */
    public abstract void writeBits(BitBuf src, long srcIndex, long length);

    public abstract void alignToByte();

    public abstract int bytesSize();

    public abstract long readableBits();

    public abstract boolean readBit();

    public abstract int readBitsToInt(int bitCount);

    public abstract long readBitsToLong(int bitCount);

    public abstract byte read8Bits();

    public abstract int read32Bits();

    public abstract long read64Bits();

    /**
     * @see BitBuf#writeIntVarint1N(int, int)
     */
    public abstract int readIntVarint1N(int max);

    /**
     * @see CodedInputStream#readRawVarint32()
     * @see BitBuf#writeIntVarint8(int)
     */
    public abstract int readIntVarint8();

    public abstract long readLongVarint8();

    public long readLongZigzagVarint8() {
        return CodedInputStream.decodeZigZag64(readLongVarint8());
    }

    // floating

    public float readFloatBits() {
        return Float.intBitsToFloat(read32Bits());
    }

    public double readDoubleBits() {
        return Double.longBitsToDouble(read64Bits());
    }

    public abstract void resetWriterIndex();

    public abstract void resetReadIndex();

    public abstract long writerIndex();

    public abstract void writerIndex(long writerIndex);

    public abstract long readerIndex();

    public abstract void readerIndex(long readerIndex);

    /**
     * Increases the current {@code readerIndex} by the specified
     * {@code length} in this buffer.
     *
     * @throws IndexOutOfBoundsException
     *         if {@code length} is greater than {@code this.readableBits}
     */
    public abstract void skipBits(long length);

    /**
     * Returns a read-only version of this buffer.
     */
    public abstract BitBuf asReadOnly();

    /**
     * Returns a slice of this buffer's sub-region. Modifying the content of
     * the returned buffer or this buffer affects each other's content while
     * they maintain separate indexes and marks.
     * This method does not modify {@code readerIndex} or {@code writerIndex} of
     * this buffer.
     */
    public abstract BitBuf slice(long index, long length);

    /**
     * Returns a buffer which shares the whole region of this buffer.
     * Modifying the content of the returned buffer or this buffer affects
     * each other's content while they maintain separate indexes.
     * This method does not modify {@code readerIndex} or {@code writerIndex} of
     * this buffer.
     */
    public abstract BitBuf duplicate();

    /**
     * Returns a copy of this buffer's readable bytes.  Modifying the content
     * of the returned buffer or this buffer does not affect each other at all.
     * This method is identical to {@code buf.copy(buf.readerIndex(), buf.readableBits())}.
     * This method does not modify {@code readerIndex} or {@code writerIndex} of
     * this buffer.
     */
    public abstract BitBuf copy();

    /**
     * Returns a copy of this buffer's sub-region.  Modifying the content of
     * the returned buffer or this buffer does not affect each other at all.
     * This method does not modify {@code readerIndex} or {@code writerIndex} of
     * this buffer.
     */
    public abstract BitBuf copy(long index, long length);

    /**
     * Returns the backing byte array of this buffer.
     *
     * @throws UnsupportedOperationException
     *         if there no accessible backing byte array
     */
    public abstract byte[] array();

    /**
     * Returns the offset of the first byte within the backing byte array of
     * this buffer.
     *
     * @throws UnsupportedOperationException
     *         if there no accessible backing byte array
     */
    public abstract int arrayOffset();

    @Override
    public abstract BitBuf retain();

    @Override
    public abstract BitBuf retain(int increment);

    @Override
    public abstract BitBuf touch();

    @Override
    public abstract BitBuf touch(Object hint);

    /**
     * Allocate new BitBuf with same allocator as source BitBif
     * @param byteCapacity initial bytes capacity
     */
    public abstract BitBuf allocate(int byteCapacity);

    /**
     * Returns {@code true} if and only if this buffer is backed by an direct buffer.
     */
    public abstract boolean isDirect();
}
