package ru.yandex.solomon.codec.compress;

import com.google.protobuf.CodedInputStream;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.TsColumn;

/**
 * Input stream with common parameters for each kind of time series
 *
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractTimeSeriesInputStream implements TimeSeriesInputStream {
    private static int[] COUNT_BIT_PER_TS_MODE = new int[] { 0, 4, 8, 12, 16, 24, 32, 64 };

    private final BitBuf in;
    private final FrameIterator frameIterator;

    private long prevStepMillis = 0;
    private boolean first = true;
    private long prevTsMillis;
    private long prevPrevTsMillis;
    private boolean millis;

    private long prevCount;
    private boolean prevMerge = false;

    public AbstractTimeSeriesInputStream(BitBuf in) {
        this.in = in;
        this.frameIterator =  new FrameIterator(in);
        nextFrame();
    }

    @Override
    public final void readPoint(int columnSet, AggrPoint point) {
        point.columnSet = columnSet;
        point.tsMillis = readTsMillis();
        readCommand();
        readValue(in, point);

        if (HasColumnSet.hasColumn(columnSet, StockpileColumn.MERGE)) {
            point.merge = prevMerge;
        }

        if (HasColumnSet.hasColumn(columnSet, StockpileColumn.COUNT)) {
            point.count = readCountInt64();
        }

        if (HasColumnSet.hasColumn(columnSet, StockpileColumn.STEP)) {
            point.stepMillis = prevStepMillis;
        }

        first = false;
        ensureFrameRead();
    }

    private void readCommand() {
        while (CommandEncoder.readCommandFlag(in)) {
            StockpileColumn column = CommandEncoder.decodeColumn(in);
            switch (column) {
                case MERGE:
                    prevMerge = !prevMerge;
                    break;
                case STEP:
                    prevStepMillis = readStepMillis();
                    break;
                default:
                    readCommand(column, in);
            }
        }
    }

    /**
     * Read rare changed parameter from command header
     * @throws IllegalStateException if command for column not supported
     */
    protected abstract void readCommand(StockpileColumn column, BitBuf stream);
    protected abstract void readValue(BitBuf stream, AggrPoint point);

    private long readStepMillis() {
        return in.readIntVarint8();
    }

    private long readTsMillis() {
        final long tsMillis;
        if (prevTsMillis == 0 || prevPrevTsMillis == 0) {
            tsMillis = in.read64Bits();
        } else {
            tsMillis = readTsMillisAfterSecond();
        }
        TsColumn.validateOrThrow(tsMillis);

        prevPrevTsMillis = prevTsMillis;
        prevTsMillis = tsMillis;

        return tsMillis;
    }

    private long readTsMillisAfterSecond() {
        int mode = in.readIntVarint1N(8);

        if (mode == 8) {
            if (millis) {
                throw new IllegalStateException("stream corrupted");
            }
            millis = true;
            mode = in.readIntVarint1N(8);
        }

        if (mode >= COUNT_BIT_PER_TS_MODE.length || mode < 0) {
            throw new IllegalStateException("stream corrupted");
        }

        long ddz = in.readBitsToLong(COUNT_BIT_PER_TS_MODE[mode]);
        long ddw = CodedInputStream.decodeZigZag64(ddz);
        long dd = millis ? ddw : ddw * 1000;

        long result = prevTsMillis + dd + prevTsMillis - prevPrevTsMillis;
        if (!TsColumn.isValid(result)) {
            throw new IllegalStateException("stream corrupted, read not valid ts: " + result);
        }

        return result;
    }

    private long readCountInt64() {
        final long count;
        if (first) {
            count = in.readLongVarint8();
        } else {
            long delta = CodedInputStream.decodeZigZag64(VarintEncoder.readVarintMode64(in));
            count = prevCount + delta;
        }

        CountColumn.validateOrThrow(count);
        prevCount = count;
        return count;
    }

    @Override
    public boolean hasNext() {
        return in.readableBits() > 0;
    }

    @Override
    public void close() {
    }

    private void ensureFrameRead() {
        if (in.readerIndex() < frameIterator.payloadIndex() + frameIterator.payloadBits()) {
            return;
        }

        nextFrame();
        resetState();
    }

    private void nextFrame() {
        if (!frameIterator.next()) {
            in.readerIndex(in.writerIndex());
            return;
        }

        in.readerIndex(frameIterator.payloadIndex());
    }

    private void resetState() {
        prevStepMillis = 0;
        first = true;
        prevTsMillis = 0;
        prevPrevTsMillis = 0;
        millis = false;
        prevCount = 0;
        prevMerge = false;
        resetAdditionalState();
    }

    protected abstract void resetAdditionalState();
}
