package ru.yandex.solomon.codec.compress;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.BitBufAllocator;
import ru.yandex.solomon.codec.compress.doubles.DoubleTimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.doubles.DoubleTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.frames.Frame;
import ru.yandex.solomon.codec.compress.frames.TimeSeriesFrameIterator;
import ru.yandex.solomon.codec.compress.frames.TimeSeriesFrameIteratorEmpty;
import ru.yandex.solomon.codec.compress.frames.TimeSeriesFrameIteratorImpl;
import ru.yandex.solomon.codec.compress.histograms.HistogramTimeSeriesInputStreamV4;
import ru.yandex.solomon.codec.compress.histograms.HistogramTimeSeriesOutputStreamV4;
import ru.yandex.solomon.codec.compress.histograms.log.LogHistogramTimeSeriesInputStreamV3;
import ru.yandex.solomon.codec.compress.histograms.log.LogHistogramTimeSeriesOutputStreamV3;
import ru.yandex.solomon.codec.compress.longs.CounterTimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.longs.CounterTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.longs.IGaugeTimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.longs.IGaugeTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.summaries.SummaryDoubleTimeSeriesInputStreamV3;
import ru.yandex.solomon.codec.compress.summaries.SummaryDoubleTimeSeriesOutputStreamV3;
import ru.yandex.solomon.codec.compress.summaries.SummaryInt64TimeSeriesInputStreamV3;
import ru.yandex.solomon.codec.compress.summaries.SummaryInt64TimeSeriesOutputStreamV3;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.LogHistogram;

import static ru.yandex.solomon.model.point.column.StockpileColumns.ensureColumnSetValid;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class CompressStreamFactory {
    private static final int DEFAULT_STREAM_CAPACITY = 3;

    private CompressStreamFactory() {
    }

    @Nonnull
    public static TimeSeriesOutputStream createOutputStream(MetricType type, int columnSetMask) {
        return createOutputStream(type, columnSetMask, DEFAULT_STREAM_CAPACITY);
    }

    @Nonnull
    public static TimeSeriesOutputStream createOutputStream(MetricType type, int columnSetMask, int capacity) {
        if (columnSetMask == StockpileColumnSet.empty.columnSetMask()) {
            return CompressStreamFactory.emptyOutputStream();
        }
        ensureColumnSetValid(type, columnSetMask);

        // temp disable init stream with pre estimate capacity
//        int pointSize = estimatePointSize(columnSetMask);
//        int bytesCapacity = Math.max(capacity * pointSize / 2, pointSize);
        BitBuf buffer = BitBufAllocator.buffer(16);
        return createOutputStream(type, columnSetMask, buffer, 0);
    }

    public static TimeSeriesOutputStream createOutputStream(MetricType type, int columnSetMask, BitBuf buffer, int records) {
        ensureColumnSetValid(type, columnSetMask);

        switch (type) {
            case DGAUGE:
                return new DoubleTimeSeriesOutputStream(buffer, records);
            case IGAUGE:
                return new IGaugeTimeSeriesOutputStream(buffer, records);
            case RATE:
            case COUNTER:
                return new CounterTimeSeriesOutputStream(buffer, records);
            case LOG_HISTOGRAM:
                return new LogHistogramTimeSeriesOutputStreamV3(buffer, records);
            case HIST:
            case HIST_RATE:
                return new HistogramTimeSeriesOutputStreamV4(buffer, records);
            case ISUMMARY:
                return new SummaryInt64TimeSeriesOutputStreamV3(buffer, records);
            case DSUMMARY:
                return new SummaryDoubleTimeSeriesOutputStreamV3(buffer, records);
            default:
                return throwNotFoundImpl(type, columnSetMask);
        }
    }

    public static TimeSeriesFrameIterator createFrameIterator(BitBuf buffer) {
        if (buffer.readableBits() == 0) {
            return TimeSeriesFrameIteratorEmpty.INSTANCE;
        }

        return new TimeSeriesFrameIteratorImpl(buffer);
    }

    @Nonnull
    public static TimeSeriesOutputStream restoreOutputStream(MetricType type, int columnSetMask, BitBuf buffer) {
        if (columnSetMask == StockpileColumnSet.empty.columnSetMask()) {
            return CompressStreamFactory.emptyOutputStream();
        }

        if (buffer.readableBits() == 0) {
            return createOutputStream(type, columnSetMask);
        }

        var fromIdx = buffer.readerIndex();
        var framesIt = createFrameIterator(buffer.duplicate());
        var frame = new Frame();
        int records = 0;
        long unclosedFrameIdx = -1;
        long lastFrameIdx = 0;
        while (framesIt.next(frame)) {
            lastFrameIdx = frame.pos;
            if (!frame.closed) {
                unclosedFrameIdx = frame.pos;
            } else {
                records += frame.records;
            }
        }

        if (unclosedFrameIdx == -1) {
            var stream = (AbstractTimeSeriesOutputStream) createOutputStream(type, columnSetMask, buffer.copy(), records);
            stream.continueLastClosedFrame(lastFrameIdx);
            return stream;
        }

        long length = buffer.writerIndex() - unclosedFrameIdx;
        var out = createOutputStream(type, columnSetMask, buffer.slice(fromIdx, unclosedFrameIdx).copy(), records);
        var point = RecyclableAggrPoint.newInstance();
        try (var in = createInputStream(type, columnSetMask, buffer.slice(unclosedFrameIdx, length))) {
            while (in.hasNext()) {
                in.readPoint(columnSetMask, point);
                out.writePoint(columnSetMask, point);
            }
        } finally {
            point.recycle();
        }

        return out;
    }

    public static TimeSeriesInputStream createInputStream(
            MetricType type,
            int columnSetMask,
            BitBuf buffer)
    {
        if (buffer.readableBits() == 0) {
            return emptyInputStream();
        }
        ensureColumnSetValid(type, columnSetMask);

        switch (type) {
            case DGAUGE:
                return new DoubleTimeSeriesInputStream(buffer);
            case IGAUGE:
                return new IGaugeTimeSeriesInputStream(buffer);
            case RATE:
            case COUNTER:
                return new CounterTimeSeriesInputStream(buffer);
            case LOG_HISTOGRAM:
                return new LogHistogramTimeSeriesInputStreamV3(buffer);
            case HIST:
            case HIST_RATE:
                return new HistogramTimeSeriesInputStreamV4(buffer);
            case ISUMMARY:
                return new SummaryInt64TimeSeriesInputStreamV3(buffer);
            case DSUMMARY:
                return new SummaryDoubleTimeSeriesInputStreamV3(buffer);
            default:
                return throwNotFoundImpl(type, columnSetMask);
        }
    }

    public static TimeSeriesInputStream emptyInputStream() {
        return EmptyTimeSeriesInputStream.INSTANCE;
    }

    public static TimeSeriesOutputStream emptyOutputStream() {
        return EmptyTimeSeriesOutputStream.INSTANCE;
    }

    public static boolean isSupported(MetricType type) {
        switch (type) {
            case DGAUGE:
            case LOG_HISTOGRAM:
            case HIST:
            case HIST_RATE:
            case ISUMMARY:
            case DSUMMARY:
            case IGAUGE:
            case RATE:
            case COUNTER:
                return true;
            default:
                return false;
        }
    }

    private static <T> T throwNotFoundImpl(MetricType type, int columnSetMask) {
        throw new IllegalArgumentException("codec for metric " + type + " ("
            + StockpileColumnSet.toString(columnSetMask)
            + ") not found");
    }

    private static int estimatePointSize(int columnSetMask) {
        int result = 0;
        if (HasColumnSet.hasColumn(columnSetMask, StockpileColumn.TS)) {
            result += Long.BYTES;
        }

        if (HasColumnSet.hasColumn(columnSetMask, StockpileColumn.VALUE)) {
            result += Double.BYTES + Long.BYTES; // num + denom
        }

        if (HasColumnSet.hasColumn(columnSetMask, StockpileColumn.MERGE)) {
            result += Byte.BYTES;
        }

        if (HasColumnSet.hasColumn(columnSetMask, StockpileColumn.COUNT)) {
            result += Integer.BYTES;
        }

        if (HasColumnSet.hasColumn(columnSetMask, StockpileColumn.STEP)) {
            result += Long.BYTES;
        }

        if (HasColumnSet.hasColumn(columnSetMask, StockpileColumn.LOG_HISTOGRAM)) {
            result += LogHistogram.LIMIT_MAX_BUCKETS_SIZE * Double.BYTES; // buckets
            result += Integer.BYTES; // bucket size
            result += Integer.BYTES; // count zeros
            result += Integer.BYTES; // start power
        }

        return result;
    }
}
