package ru.yandex.solomon.codec.compress;

import java.util.function.Consumer;

import ru.yandex.solomon.codec.bits.BitArray;
import ru.yandex.solomon.codec.bits.BitBuf;

/**
 * @author Vladimir Gordiychuk
 */
public class FrameEncoder {
    static final int FRAME_HEADER_SIZE = Integer.SIZE + Byte.SIZE;
    static final int FRAME_FOOTER_SIZE = Integer.SIZE;

    /**
     * @return index of frame start
     */
    public static long initFrame(BitBuf buffer) {
        long frameStartIdx = buffer.writerIndex();
        buffer.write32Bits(0);
        buffer.write8Bits(0);
        return frameStartIdx;
    }

    public static void continueFrame(BitBuf buffer, long frameStartIdx) {
        long idx = buffer.writerIndex();
        buffer.writerIndex(frameStartIdx);
        initFrame(buffer);
        buffer.writerIndex(idx);
    }


    public static void finishFrame(BitBuf buffer, long frameStartIdx, Consumer<BitBuf> footerFn) {
        int usedBitsForLastByte = BitArray.mod8(buffer.writerIndex());
        buffer.alignToByte();
        long endPointsIdx = buffer.writerIndex();
        {
            // header
            int frameBytes = BitArray.div8(endPointsIdx - frameStartIdx - FRAME_HEADER_SIZE);
            buffer.writerIndex(frameStartIdx);
            buffer.write32Bits(frameBytes);
            buffer.write8Bits(usedBitsForLastByte);
            buffer.writerIndex(endPointsIdx);
        }

        {
            // footer
            buffer.write32Bits(0); // size of state unknown right now
            footerFn.accept(buffer);
            buffer.alignToByte();
            long endStateIdx = buffer.writerIndex();
            buffer.writerIndex(endPointsIdx);
            buffer.write32Bits(BitArray.div8(endStateIdx - endPointsIdx - FRAME_FOOTER_SIZE));
            buffer.writerIndex(endStateIdx);
        }
    }
}
