package ru.yandex.solomon.codec.compress;

import ru.yandex.misc.lang.ByteUtils;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
public final class GorillaEncoder {
    private GorillaEncoder() {
    }

    public static void write(BitBuf stream, State state, double value) {
        long rawValue = Double.doubleToRawLongBits(value);
        long rawPrev = Double.doubleToRawLongBits(state.prev);
        if (rawValue == rawPrev) {
            // 2. If XOR with the previous is zero (same value), store single ‘0’ bit
            stream.writeBit(false);
            return;
        }

        long xor = rawValue ^ rawPrev;

        int lz = Long.numberOfLeadingZeros(xor);
        int tz = Long.numberOfTrailingZeros(xor);

        if (lz > 31) {
            lz = 31;
        }

        // 3. When XOR is non-zero, calculate the number of leading
        // and trailing zeros in the XOR, store bit ‘1’ followed
        // by either a) or b):
        stream.writeBit(true);

        final int meaningfulBits;
        if (lz >= state.prevLeadingZeros && tz >= state.prevTrailingZeros && (state.prevLeadingZeros != 0 || state.prevTrailingZeros != 0)) {
            // (a) (Control bit ‘0’) If the block of meaningful bits
            // falls within the block of previous meaningful bits,
            // i.e., there are at least as many leading zeros and
            // as many trailing zeros as with the previous value,
            // use that information for the block position and
            // just store the meaningful XORed value.

            tz = state.prevTrailingZeros;
            lz = state.prevLeadingZeros;
            meaningfulBits = 64 - state.prevTrailingZeros - state.prevLeadingZeros;

            stream.writeBit(false);
        } else {
            // (b) (Control bit ‘1’) Store the length of the number
            // of leading zeros in the next 5 bits, then store the
            // length of the meaningful XORed value in the next
            // 6 bits. Finally store the meaningful bits of the
            // XORed value.

            meaningfulBits = 64 - lz - tz;

            stream.writeBit(true);
            stream.writeBits(lz, 5);
            stream.writeBits(meaningfulBits - 1, 6);
        }

        long meaningfulXor = xor >>> tz;
        stream.writeBits(meaningfulXor, meaningfulBits);

        state.prev = value;
        state.prevTrailingZeros = ByteUtils.toByteExact(tz);
        state.prevLeadingZeros = ByteUtils.toByteExact(lz);
    }

    public static double read(BitBuf stream, State state) {
        if (!stream.readBit()) {
            return state.prev;
        }

        final int lz;
        final int tz;
        final int meaningfulBits;
        if (!stream.readBit() && (state.prevLeadingZeros != 0 || state.prevTrailingZeros != 0)) {
            lz = state.prevLeadingZeros;
            tz = state.prevTrailingZeros;
            meaningfulBits = 64 - lz - tz;
        } else {
            lz = stream.readBitsToInt(5);
            meaningfulBits = stream.readBitsToInt(6) + 1;
            tz = 64 - meaningfulBits - lz;
        }

        long meaningfulXor = stream.readBitsToLong(meaningfulBits);
        long xor = meaningfulXor << tz;

        long l = xor ^ Double.doubleToRawLongBits(state.prev);
        double d = Double.longBitsToDouble(l);

        state.prev = d;
        state.prevTrailingZeros = ByteUtils.toByteExact(tz);
        state.prevLeadingZeros = ByteUtils.toByteExact(lz);

        return d;
    }

    public static class State implements MemMeasurable {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(State.class);

        public double prev = 0;
        public byte prevTrailingZeros = 0;
        public byte prevLeadingZeros = 0;

        public State() {
        }

        public State(State state) {
            this.prev = state.prev;
            this.prevTrailingZeros = state.prevTrailingZeros;
            this.prevLeadingZeros = state.prevLeadingZeros;
        }

        public void reset() {
            this.prev = 0;
            this.prevTrailingZeros = 0;
            this.prevLeadingZeros = 0;
        }

        public double getPrev() {
            return prev;
        }

        @Override
        public long memorySizeIncludingSelf() {
            return SELF_SIZE;
        }

        public void dumpAndReset(BitBuf buffer) {
            buffer.writeDoubleBits(prev);
            prev = 0;
            buffer.write8Bits(prevTrailingZeros);
            prevTrailingZeros = 0;
            buffer.write8Bits(prevLeadingZeros);
            prevLeadingZeros = 0;
        }

        public void restore(BitBuf buffer) {
            prev = buffer.readDoubleBits();
            prevTrailingZeros = buffer.read8Bits();
            prevLeadingZeros = buffer.read8Bits();
        }
    }
}
