package ru.yandex.solomon.codec.compress.doubles;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.ByteUtils;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesInputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DoubleTimeSeriesInputStream extends AbstractTimeSeriesInputStream {
    private long prevValueDenom = ValueColumn.DEFAULT_DENOM;
    private double prevValueNum = 0;
    private byte prevValueNumTrailingZeros = 0;
    private byte prevValueNumLeadingZeros = 0;

    public DoubleTimeSeriesInputStream(BitBuf in) {
        super(in);
    }

    @Override
    protected void readCommand(StockpileColumn column, BitBuf stream) {
        if (column != StockpileColumn.VALUE) {
            throw new IllegalStateException("unknown command for column: " + column);
        }

        prevValueDenom = DenomEncoder.read(stream);
    }

    @Override
    protected void readValue(BitBuf stream, AggrPoint point) {
        point.valueNum = readValueNum(stream);
        point.valueDenom = prevValueDenom;
    }

    private double readValueNum(BitBuf stream) {
        if (!stream.readBit()) {
            return prevValueNum;
        }

        final int lz;
        final int tz;
        final int meaningfulBits;
        if (!stream.readBit() && (prevValueNumLeadingZeros != 0 || prevValueNumTrailingZeros != 0)) {
            lz = prevValueNumLeadingZeros;
            tz = prevValueNumTrailingZeros;
            meaningfulBits = 64 - lz - tz;
        } else {
            lz = stream.readBitsToInt(5);
            meaningfulBits = stream.readBitsToInt(6) + 1;
            tz = 64 - meaningfulBits - lz;
        }

        long meaningfulXor = stream.readBitsToLong(meaningfulBits);
        long xor = meaningfulXor << tz;

        long l = xor ^ Double.doubleToRawLongBits(prevValueNum);
        double d = Double.longBitsToDouble(l);

        prevValueNum = d;
        prevValueNumTrailingZeros = ByteUtils.toByteExact(tz);
        prevValueNumLeadingZeros = ByteUtils.toByteExact(lz);

        return d;
    }

    @Override
    protected void resetAdditionalState() {
        prevValueDenom = ValueColumn.DEFAULT_DENOM;
        prevValueNum = 0;
        prevValueNumTrailingZeros = 0;
        prevValueNumLeadingZeros = 0;
    }
}
