package ru.yandex.solomon.codec.compress.doubles;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.ByteUtils;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.CommandEncoder;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DoubleTimeSeriesOutputStream extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(DoubleTimeSeriesOutputStream.class);

    private long prevValueDenom;
    private long prevValueNumRaw;
    private byte prevValueNumTrailingZeros;
    private byte prevValueNumLeadingZeros;

    public DoubleTimeSeriesOutputStream(BitBuf out, int records) {
        super(out, records);
        this.prevValueDenom = ValueColumn.DEFAULT_DENOM;
        this.prevValueNumRaw = 0;
        this.prevValueNumLeadingZeros = 0;
        this.prevValueNumLeadingZeros = 0;
    }

    public DoubleTimeSeriesOutputStream(DoubleTimeSeriesOutputStream copy) {
        super(copy);
        this.prevValueDenom = copy.prevValueDenom;
        this.prevValueNumRaw = copy.prevValueNumRaw;
        this.prevValueNumLeadingZeros = copy.prevValueNumLeadingZeros;
        this.prevValueNumTrailingZeros = copy.prevValueNumTrailingZeros;
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        writeDenomCommand(stream, point.valueDenom);
    }

    @Override
    protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {
        writeValueNum(stream, point.valueNum);
    }

    private void writeDenomCommand(BitBuf stream, long denom) {
        if (prevValueDenom != denom) {
            CommandEncoder.encodeCommandPrefix(stream, StockpileColumn.VALUE);
            DenomEncoder.write(stream, denom);
            prevValueDenom = denom;
        }
    }

    /**
     * Gorilla white paper implementation http://www.vldb.org/pvldb/vol8/p1816-teller.pdf
     * The same logic applies during encode log histogram
     */
    private void writeValueNum(BitBuf stream, double value) {
        long rawValue = Double.doubleToRawLongBits(value);
        if (rawValue == prevValueNumRaw) {
            // 2. If XOR with the previous is zero (same value), store single ‘0’ bit
            stream.writeBit(false);
            return;
        }

        long xor = rawValue ^ prevValueNumRaw;

        int lz = Long.numberOfLeadingZeros(xor);
        int tz = Long.numberOfTrailingZeros(xor);

        if (lz > 31) {
            lz = 31;
        }

        final int meaningfulBits;

        // 3. When XOR is non-zero, calculate the number of leading
        // and trailing zeros in the XOR, store bit ‘1’ followed
        // by either a) or b):
        stream.writeBit(true);

        if (lz >= prevValueNumLeadingZeros && tz >= prevValueNumTrailingZeros && (prevValueNumLeadingZeros != 0 || prevValueNumTrailingZeros != 0)) {
            // (a) (Control bit ‘0’) If the block of meaningful bits
            // falls within the block of previous meaningful bits,
            // i.e., there are at least as many leading zeros and
            // as many trailing zeros as with the previous value,
            // use that information for the block position and
            // just store the meaningful XORed value.

            tz = prevValueNumTrailingZeros;
            lz = prevValueNumLeadingZeros;
            meaningfulBits = 64 - prevValueNumTrailingZeros - prevValueNumLeadingZeros;

            stream.writeBit(false);
        } else {
            // (b) (Control bit ‘1’) Store the length of the number
            // of leading zeros in the next 5 bits, then store the
            // length of the meaningful XORed value in the next
            // 6 bits. Finally store the meaningful bits of the
            // XORed value.

            meaningfulBits = 64 - lz - tz;

            stream.writeBit(true);
            stream.writeBits(lz, 5);
            stream.writeBits(meaningfulBits - 1, 6);
        }

        long meaningfulXor = xor >>> tz;
        stream.writeBits(meaningfulXor, meaningfulBits);

        prevValueNumRaw = rawValue;
        prevValueNumTrailingZeros = ByteUtils.toByteExact(tz);
        prevValueNumLeadingZeros = ByteUtils.toByteExact(lz);
    }

    @Override
    protected long memorySelfSize() {
        return SELF_SIZE;
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new DoubleTimeSeriesOutputStream(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        DenomEncoder.write(buffer, prevValueDenom);
        prevValueDenom = ValueColumn.DEFAULT_DENOM;

        buffer.write64Bits(prevValueNumRaw);
        prevValueNumRaw = 0;

        buffer.write8Bits(prevValueNumLeadingZeros);
        prevValueNumLeadingZeros = 0;

        buffer.write8Bits(prevValueNumTrailingZeros);
        prevValueNumTrailingZeros = 0;
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        prevValueDenom = DenomEncoder.read(buffer);
        prevValueNumRaw = buffer.read64Bits();
        ValueColumn.validateOrThrow(Double.longBitsToDouble(prevValueNumRaw), prevValueDenom);

        prevValueNumLeadingZeros = buffer.read8Bits();
        prevValueNumTrailingZeros = buffer.read8Bits();
        if (prevValueNumLeadingZeros > 64 || prevValueNumTrailingZeros > 64) {
            throw new CorruptedBinaryDataRuntimeException("Invalid leading/trailing zeros");
        }
    }
}
