package ru.yandex.solomon.codec.compress.histograms;

import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedOutputStream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.CommandEncoder;
import ru.yandex.solomon.codec.compress.GorillaEncoder;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.codec.compress.doubles.DenomEncoder;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.type.Histogram;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class HistogramTimeSeriesOutputStreamV4 extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(HistogramTimeSeriesOutputStreamV4.class);
    private double[] prevBounds = Cf.DoubleArray.emptyArray();
    private byte[] prevBoundsTrailingZeros = Cf.ByteArray.emptyArray();
    private byte[] prevBoundsLeadingZeros = Cf.ByteArray.emptyArray();
    private long[] prevBuckets = Cf.LongArray.emptyArray();
    private int size = 0;
    private long prevDenom = 0;

    public HistogramTimeSeriesOutputStreamV4(BitBuf out, int records) {
        super(out, records);
    }

    public HistogramTimeSeriesOutputStreamV4(HistogramTimeSeriesOutputStreamV4 copy) {
        super(copy);
        this.prevBounds = copy.prevBounds.clone();
        this.prevBoundsTrailingZeros = copy.prevBoundsTrailingZeros.clone();
        this.prevBoundsLeadingZeros = copy.prevBoundsLeadingZeros.clone();
        this.prevBuckets = copy.prevBuckets.clone();
        this.size = copy.size;
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        var histogram = point.histogram;
        // Bounds for histogram should not change very often at least it's true for solomon
        // see ru.yandex.monlib.metrics.histogram.Histograms
        if (histogram == null) {
            if (size == 0) {
                return;
            }
            size = 0;
            stream.writeIntVarint8(size);
            prevDenom = 0;
            DenomEncoder.write(stream, prevDenom);
        } else if (!boundsEquals(histogram) || prevDenom != histogram.getDenom()) {
            CommandEncoder.encodeCommandPrefix(stream, StockpileColumn.HISTOGRAM);
            if (size < histogram.count()) {
                prevBounds = Arrays.copyOf(prevBounds, histogram.count());
                prevBoundsLeadingZeros = Arrays.copyOf(prevBoundsLeadingZeros, histogram.count());
                prevBoundsTrailingZeros = Arrays.copyOf(prevBoundsTrailingZeros, histogram.count());
                prevBuckets = Arrays.copyOf(prevBuckets, histogram.count());
            }
            this.size = histogram.count();
            stream.writeIntVarint8(size);
            GorillaEncoder.State state = new GorillaEncoder.State();
            for (int index = 0; index < histogram.count(); index++) {
                state.prev = prevBounds[index];
                state.prevLeadingZeros = prevBoundsLeadingZeros[index];
                state.prevTrailingZeros = prevBoundsTrailingZeros[index];

                double upperBounds = histogram.upperBound(index);
                GorillaEncoder.write(stream, state, upperBounds);

                prevBounds[index] = state.prev;
                prevBoundsLeadingZeros[index] = state.prevLeadingZeros;
                prevBoundsTrailingZeros[index] = state.prevTrailingZeros;
            }
            this.prevDenom = histogram.getDenom();
            DenomEncoder.write(stream, prevDenom);
        }
    }

    private boolean boundsEquals(Histogram snapshot) {
        if (snapshot.count() != size) {
            return false;
        }

        for (int index = 0; index < size; index++) {
            if (snapshot.upperBound(index) != prevBounds[index]) {
                return false;
            }
        }

        return true;
    }

    @Override
    protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {
        if (point.histogram == null) {
            return;
        }
        var histogram = point.histogram;
        for (int index = 0; index < histogram.count(); index++) {
            long prevValue = prevBuckets[index];
            long value = histogram.value(index);
            long delta = value - prevValue;
            VarintEncoder.writeVarintMode64(stream, CodedOutputStream.encodeZigZag64(delta));
            prevBuckets[index] = value;
        }
    }

    @Override
    protected long memorySelfSize() {
        long size = SELF_SIZE;
        size += MemoryCounter.arrayObjectSize(prevBounds);
        size += MemoryCounter.arrayObjectSize(prevBoundsLeadingZeros);
        size += MemoryCounter.arrayObjectSize(prevBoundsTrailingZeros);
        size += MemoryCounter.arrayObjectSize(prevBuckets);
        return size;
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new HistogramTimeSeriesOutputStreamV4(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        VarintEncoder.writeVarintMode32(buffer, size);
        for (int index = 0; index < prevBounds.length; index++) {
            buffer.writeDoubleBits(prevBounds[index]);
            prevBounds[index] = 0;

            buffer.write8Bits(prevBoundsTrailingZeros[index]);
            prevBoundsTrailingZeros[index] = 0;

            buffer.write8Bits(prevBoundsLeadingZeros[index]);
            prevBoundsLeadingZeros[index] = 0;

            VarintEncoder.writeVarintMode64(buffer, prevBuckets[index]);
            prevBuckets[index] = 0;
        }
        VarintEncoder.writeVarintMode32(buffer, size);
        size = 0;
        DenomEncoder.write(buffer, prevDenom);
        prevDenom = 0;
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        int arraySize = VarintEncoder.readVarintMode32(buffer);
        prevBounds = new double[arraySize];
        prevBoundsTrailingZeros = new byte[arraySize];
        prevBoundsLeadingZeros = new byte[arraySize];
        prevBuckets = new long[arraySize];

        for (int index = 0; index < arraySize; index++) {
            prevBounds[index] = buffer.readDoubleBits();
            prevBoundsTrailingZeros[index] = buffer.read8Bits();
            prevBoundsLeadingZeros[index] = buffer.read8Bits();
            prevBuckets[index] = VarintEncoder.readVarintMode64(buffer);
        }
        size = VarintEncoder.readVarintMode32(buffer);
        prevDenom = DenomEncoder.read(buffer);
    }
}
