package ru.yandex.solomon.codec.compress.histograms.log;

import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.GorillaEncoder;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@SuppressWarnings("Duplicates")
@ParametersAreNonnullByDefault
public class LogHistogramTimeSeriesInputStreamV3 extends AbstractTimeSeriesInputStream {
    private int prevStartPower;
    private long prevCountZero;
    private int prevCountBuckets;
    private int prevMaxBuckets = LogHistogram.DEFAULT_MAX_BUCKET_SIZE;
    private double prevBase = LogHistogram.DEFAULT_BASE;
    private double[] prevBuckets = Cf.DoubleArray.emptyArray();
    private byte[] prevBucketValueTrailingZeros = Cf.ByteArray.emptyArray();
    private byte[] prevBucketValueLeadingZeros = Cf.ByteArray.emptyArray();
    private final GorillaEncoder.State state = new GorillaEncoder.State();

    public LogHistogramTimeSeriesInputStreamV3(BitBuf in) {
        super(in);
    }

    @Override
    protected void readCommand(StockpileColumn column, BitBuf stream) {
        prevMaxBuckets = stream.readBitsToInt(LogHistogramEncodeConstants.MAX_BUCKET_SIZE_IN_BITS);
        prevBase = stream.readDoubleBits();
    }

    @Override
    protected void readValue(BitBuf stream, AggrPoint point) {
        readStartPower(stream);
        readCountZero(stream);
        readSizeBuckets(stream);

        var histogram = LogHistogram.orNew(point.logHistogram)
            .setMaxBucketsSize(prevMaxBuckets)
            .setBase(prevBase)
            .setStartPower(prevStartPower)
            .setCountZero(prevCountZero);

        ensureBucketsCapacity(prevCountBuckets);
        for (int index = 0; index < prevCountBuckets; index++) {
            state.prev = prevBuckets[index];
            state.prevTrailingZeros = prevBucketValueTrailingZeros[index];
            state.prevLeadingZeros = prevBucketValueLeadingZeros[index];

            GorillaEncoder.read(stream, state);
            histogram.addBucket(state.prev);

            prevBuckets[index] = state.prev;
            prevBucketValueTrailingZeros[index] = state.prevTrailingZeros;
            prevBucketValueLeadingZeros[index] = state.prevLeadingZeros;
        }

        point.logHistogram = histogram;
    }

    private void readSizeBuckets(BitBuf os) {
        prevCountBuckets += CodedInputStream.decodeZigZag32(VarintEncoder.readVarintMode32(os));
    }

    private void readStartPower(BitBuf os) {
        prevStartPower += CodedInputStream.decodeZigZag32(VarintEncoder.readVarintMode32(os));
    }

    private void readCountZero(BitBuf os) {
        prevCountZero += CodedInputStream.decodeZigZag64(VarintEncoder.readVarintMode64(os));
    }

    private void ensureBucketsCapacity(int expectedSize) {
        if (prevBuckets.length < expectedSize) {
            prevBuckets = Arrays.copyOfRange(prevBuckets, 0, expectedSize);
            prevBucketValueTrailingZeros = Arrays.copyOfRange(prevBucketValueTrailingZeros, 0, expectedSize);
            prevBucketValueLeadingZeros = Arrays.copyOfRange(prevBucketValueLeadingZeros, 0, expectedSize);
        }
    }

    @Override
    protected void resetAdditionalState() {
        prevStartPower = 0;
        prevCountZero = 0;
        prevCountBuckets = 0;
        prevMaxBuckets = LogHistogram.DEFAULT_MAX_BUCKET_SIZE;
        prevBase = LogHistogram.DEFAULT_BASE;
        Arrays.fill(prevBuckets, 0);
        Arrays.fill(prevBucketValueTrailingZeros, (byte) 0);
        Arrays.fill(prevBucketValueLeadingZeros, (byte) 0);
        state.reset();
    }
}
