package ru.yandex.solomon.codec.compress.histograms.log;

import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedOutputStream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.CommandEncoder;
import ru.yandex.solomon.codec.compress.GorillaEncoder;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@SuppressWarnings("Duplicates")
@ParametersAreNonnullByDefault
public class LogHistogramTimeSeriesOutputStreamV3 extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(LogHistogramTimeSeriesOutputStreamV3.class);

    private int prevStartPower;
    private long prevCountZero;
    private int prevCountBuckets;
    private int prevMaxBuckets = LogHistogram.DEFAULT_MAX_BUCKET_SIZE;
    private double prevBase = LogHistogram.DEFAULT_BASE;
    private double[] prevBuckets = Cf.DoubleArray.emptyArray();
    private byte[] prevBucketValueTrailingZeros = Cf.ByteArray.emptyArray();
    private byte[] prevBucketValueLeadingZeros = Cf.ByteArray.emptyArray();
    private GorillaEncoder.State state = new GorillaEncoder.State();

    public LogHistogramTimeSeriesOutputStreamV3(BitBuf out, int records) {
        super(out, records);
    }

    public LogHistogramTimeSeriesOutputStreamV3(LogHistogramTimeSeriesOutputStreamV3 copy) {
        super(copy);
        this.prevStartPower = copy.prevStartPower;
        this.prevCountZero = copy.prevCountZero;
        this.prevCountBuckets = copy.prevCountBuckets;
        this.prevMaxBuckets = copy.prevMaxBuckets;
        this.prevBase = copy.prevBase;
        this.prevBuckets = copy.prevBuckets.clone();
        this.prevBucketValueTrailingZeros = copy.prevBucketValueTrailingZeros.clone();
        this.prevBucketValueLeadingZeros = copy.prevBucketValueLeadingZeros.clone();
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        LogHistogram histogram = point.logHistogram;

        // base and max bucket size changes not so often, and we can write it together when in changed without any problem
        if (prevMaxBuckets != histogram.getMaxBucketsSize() || Double.compare(prevBase, histogram.getBase()) != 0) {
            CommandEncoder.encodeCommandPrefix(stream, StockpileColumn.LOG_HISTOGRAM);

            prevMaxBuckets = histogram.getMaxBucketsSize();
            stream.writeBits(prevMaxBuckets, LogHistogramEncodeConstants.MAX_BUCKET_SIZE_IN_BITS);

            prevBase = histogram.getBase();
            stream.writeDoubleBits(prevBase);
        }
    }

    @Override
    protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {
        LogHistogram histogram = point.logHistogram;
        writeStartPower(stream, histogram.getStartPower());
        writeCountZero(stream, histogram.getCountZero());
        writeSizeBuckets(stream, histogram.countBucket());

        ensureBucketsCapacity(histogram.countBucket());
        for (int index = 0; index < histogram.countBucket(); index++) {
            state.prev = prevBuckets[index];
            state.prevLeadingZeros = prevBucketValueLeadingZeros[index];
            state.prevTrailingZeros = prevBucketValueTrailingZeros[index];

            GorillaEncoder.write(stream, state, histogram.getBucketValue(index));

            prevBuckets[index] = state.prev;
            prevBucketValueLeadingZeros[index] = state.prevLeadingZeros;
            prevBucketValueTrailingZeros[index] = state.prevTrailingZeros;
        }
    }

    private void writeSizeBuckets(BitBuf os, int countBucket) {
        int delta = countBucket - prevCountBuckets;
        VarintEncoder.writeVarintMode32(os, CodedOutputStream.encodeZigZag32(delta));
        this.prevCountBuckets = countBucket;
    }

    private void writeStartPower(BitBuf os, int current) {
        int delta = current - prevStartPower;
        VarintEncoder.writeVarintMode32(os, CodedOutputStream.encodeZigZag32(delta));
        this.prevStartPower = current;
    }

    private void writeCountZero(BitBuf os, long current) {
        long delta = current - prevCountZero;
        VarintEncoder.writeVarintMode64(os, CodedOutputStream.encodeZigZag64(delta));
        this.prevCountZero = current;
    }

    private void ensureBucketsCapacity(int expectedSize) {
        if (prevBuckets.length < expectedSize) {
            prevBuckets = Arrays.copyOfRange(prevBuckets, 0, expectedSize);
            prevBucketValueTrailingZeros = Arrays.copyOfRange(prevBucketValueTrailingZeros, 0, expectedSize);
            prevBucketValueLeadingZeros = Arrays.copyOfRange(prevBucketValueLeadingZeros, 0, expectedSize);
        }
    }

    @Override
    protected long memorySelfSize() {
        return SELF_SIZE
            + MemoryCounter.arrayObjectSize(prevBuckets)
            + MemoryCounter.arrayObjectSize(prevBucketValueTrailingZeros)
            + MemoryCounter.arrayObjectSize(prevBucketValueLeadingZeros)
            + state.memorySizeIncludingSelf();
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new LogHistogramTimeSeriesOutputStreamV3(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        VarintEncoder.writeVarintMode32(buffer, prevStartPower);
        prevStartPower = 0;

        // TODO: change format
        VarintEncoder.writeVarintMode64(buffer, prevCountZero);
        prevCountZero = 0;

        VarintEncoder.writeVarintMode32(buffer, prevCountBuckets);
        prevCountBuckets = 0;

        VarintEncoder.writeVarintMode32(buffer, prevMaxBuckets);
        prevMaxBuckets = LogHistogram.DEFAULT_MAX_BUCKET_SIZE;

        buffer.writeDoubleBits(prevBase);
        prevBase = LogHistogram.DEFAULT_BASE;

        VarintEncoder.writeVarintMode32(buffer, prevBuckets.length);
        for (int index = 0; index < prevBuckets.length; index++) {
            buffer.writeDoubleBits(prevBuckets[index]);
            prevBuckets[index] = 0;

            buffer.write8Bits(prevBucketValueTrailingZeros[index]);
            prevBucketValueTrailingZeros[index] = 0;

            buffer.write8Bits(prevBucketValueLeadingZeros[index]);
            prevBucketValueLeadingZeros[index] = 0;
        }
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        prevStartPower = VarintEncoder.readVarintMode32(buffer);
        prevCountZero = VarintEncoder.readVarintMode64(buffer);
        prevCountBuckets = VarintEncoder.readVarintMode32(buffer);
        prevMaxBuckets = VarintEncoder.readVarintMode32(buffer);
        prevBase = buffer.readDoubleBits();

        int size = VarintEncoder.readVarintMode32(buffer);
        prevBuckets = new double[size];
        prevBucketValueTrailingZeros = new byte[size];
        prevBucketValueLeadingZeros = new byte[size];
        for (int index = 0; index < size; index++) {
            prevBuckets[index] = buffer.readDoubleBits();
            prevBucketValueTrailingZeros[index] = buffer.read8Bits();
            prevBucketValueLeadingZeros[index] = buffer.read8Bits();
        }
    }
}
