package ru.yandex.solomon.codec.compress.longs;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedOutputStream;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.LongValueColumn;

/**
 * Delta-delta value encoding
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class CounterTimeSeriesOutputStream extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(CounterTimeSeriesOutputStream.class);

    private long prevValue;
    private long prevValueDelta;

    public CounterTimeSeriesOutputStream(BitBuf out, int records) {
        super(out, records);
    }

    public CounterTimeSeriesOutputStream(CounterTimeSeriesOutputStream copy) {
        super(copy);
        this.prevValue = copy.prevValue;
        this.prevValueDelta = copy.prevValueDelta;
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        // no op
    }

    @Override
    protected void writeValue(BitBuf out, int columnSet, AggrPointData point) {
        long value = point.longValue;
        long delta = value - prevValue;
        long deltaOfDelta = delta - prevValueDelta;
        VarintEncoder.writeVarintMode64(out, CodedOutputStream.encodeZigZag64(deltaOfDelta));
        this.prevValueDelta = delta;
        this.prevValue = value;
    }

    @Override
    protected long memorySelfSize() {
        return SELF_SIZE;
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new CounterTimeSeriesOutputStream(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        VarintEncoder.writeVarintMode64(buffer, prevValue);
        prevValue = 0;
        VarintEncoder.writeVarintMode64(buffer, prevValueDelta);
        prevValueDelta = 0;
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        prevValue = VarintEncoder.readVarintMode64(buffer);
        LongValueColumn.validateOrThrow(prevValue);
        prevValueDelta = VarintEncoder.readVarintMode64(buffer);
    }
}
