package ru.yandex.solomon.codec.compress.longs;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedOutputStream;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.LongValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class IGaugeTimeSeriesOutputStream extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(IGaugeTimeSeriesOutputStream.class);
    private long prevLongValue = LongValueColumn.DEFAULT_VALUE;

    public IGaugeTimeSeriesOutputStream(BitBuf out, int records) {
        super(out, records);
    }

    public IGaugeTimeSeriesOutputStream(IGaugeTimeSeriesOutputStream copy) {
        super(copy);
        this.prevLongValue = copy.prevLongValue;
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        // no op
    }

    @Override
    protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {
        long value = point.longValue;
        long delta = value - prevLongValue;
        VarintEncoder.writeVarintMode64(stream, CodedOutputStream.encodeZigZag64(delta));
        prevLongValue = value;
    }

    @Override
    protected long memorySelfSize() {
        return SELF_SIZE;
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new IGaugeTimeSeriesOutputStream(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        VarintEncoder.writeVarintMode64(buffer, prevLongValue);
        prevLongValue = LongValueColumn.DEFAULT_VALUE;
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        prevLongValue = VarintEncoder.readVarintMode64(buffer);
        LongValueColumn.validateOrThrow(prevLongValue);
    }
}
