package ru.yandex.solomon.codec.compress.summaries;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedOutputStream;

import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.CommandEncoder;
import ru.yandex.solomon.codec.compress.GorillaEncoder;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;

import static ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot.EMPTY;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SummaryDoubleTimeSeriesOutputStreamV3 extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SummaryDoubleTimeSeriesOutputStreamV3.class);

    private int mask = 0;
    private long prevCount = 0;
    private GorillaEncoder.State prevSum = new GorillaEncoder.State();
    private GorillaEncoder.State prevMin = new GorillaEncoder.State();
    private GorillaEncoder.State prevMax = new GorillaEncoder.State();
    private GorillaEncoder.State prevLast = new GorillaEncoder.State();

    public SummaryDoubleTimeSeriesOutputStreamV3(BitBuf out, int records) {
        super(out, records);
    }

    public SummaryDoubleTimeSeriesOutputStreamV3(SummaryDoubleTimeSeriesOutputStreamV3 copy) {
        super(copy);
        this.mask = copy.mask;
        this.prevCount = copy.prevCount;
        this.prevSum = new GorillaEncoder.State(copy.prevSum);
        this.prevMin = new GorillaEncoder.State(copy.prevMin);
        this.prevMax = new GorillaEncoder.State(copy.prevMax);
        this.prevLast = new GorillaEncoder.State(copy.prevLast);
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        SummaryDoubleSnapshot summary = point.summaryDouble;

        int mask = 0;
        if (summary.getCount() != EMPTY.getCount()) {
            mask |= SummaryColumn.COUNT.getMask();
        }

        if (summary.getSum() != EMPTY.getSum()) {
            mask |= SummaryColumn.SUM.getMask();
        }

        if (summary.getMin() != EMPTY.getMin()) {
            mask |= SummaryColumn.MIN.getMask();
        }

        if (summary.getMax() != EMPTY.getMax()) {
            mask |= SummaryColumn.MAX.getMask();
        }

        if (summary.getLast() != EMPTY.getLast()) {
            mask |= SummaryColumn.LAST.getMask();
        }

        if ((this.mask & mask) != mask) {
            this.mask |= mask;
            CommandEncoder.encodeCommandPrefix(stream, StockpileColumn.DSUMMARY);
            stream.writeBits(this.mask, 5);
        }
    }

    @Override
    protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {
        SummaryDoubleSnapshot summary = point.summaryDouble;
        if (SummaryColumn.COUNT.present(mask)) {
            long delta = summary.getCount() - prevCount;
            VarintEncoder.writeVarintMode64(stream, CodedOutputStream.encodeZigZag64(delta));
            prevCount = summary.getCount();
        }

        if (SummaryColumn.SUM.present(mask)) {
            GorillaEncoder.write(stream, prevSum, summary.getSum());
        }

        if (SummaryColumn.MIN.present(mask)) {
            GorillaEncoder.write(stream, prevMin, summary.getMin());
        }

        if (SummaryColumn.MAX.present(mask)) {
            GorillaEncoder.write(stream, prevMax, summary.getMax());
        }

        if (SummaryColumn.LAST.present(mask)) {
            GorillaEncoder.write(stream, prevLast, summary.getLast());
        }
    }

    @Override
    protected long memorySelfSize() {
        return SELF_SIZE
            + prevSum.memorySizeIncludingSelf()
            + prevMin.memorySizeIncludingSelf()
            + prevMax.memorySizeIncludingSelf()
            + prevLast.memorySizeIncludingSelf();
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new SummaryDoubleTimeSeriesOutputStreamV3(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        buffer.writeBits(this.mask, 5);
        mask = 0;

        VarintEncoder.writeVarintMode64(buffer, prevCount);
        prevCount = 0;
        prevSum.dumpAndReset(buffer);
        prevMin.dumpAndReset(buffer);
        prevMax.dumpAndReset(buffer);
        prevLast.dumpAndReset(buffer);
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        mask = buffer.readBitsToInt(5);
        prevCount = VarintEncoder.readVarintMode64(buffer);
        prevSum.restore(buffer);
        prevMin.restore(buffer);
        prevMax.restore(buffer);
        prevLast.restore(buffer);
    }
}

