package ru.yandex.solomon.codec.compress.summaries;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesInputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SummaryInt64TimeSeriesInputStreamV3 extends AbstractTimeSeriesInputStream {
    private int mask;
    private long prevMax;
    private long prevMin;
    private long prevCount;
    private long prevSum;
    private long prevLast;

    public SummaryInt64TimeSeriesInputStreamV3(BitBuf in) {
        super(in);
    }

    @Override
    protected void readCommand(StockpileColumn column, BitBuf stream) {
        mask = stream.readBitsToInt(5);
    }

    @Override
    protected void readValue(BitBuf stream, AggrPoint point) {
        if (SummaryColumn.COUNT.present(mask)) {
            prevCount += readDelta(stream);
        }

        if (SummaryColumn.SUM.present(mask)) {
            prevSum += readDelta(stream);
        }

        long min;
        if (SummaryColumn.MIN.present(mask)) {
            prevMin += readDelta(stream);
            min = prevMin;
        } else {
            min = ImmutableSummaryInt64Snapshot.EMPTY.getMin();
        }

        long max;
        if (SummaryColumn.MAX.present(mask)) {
            prevMax += readDelta(stream);
            max = prevMax;
        } else {
            max = ImmutableSummaryInt64Snapshot.EMPTY.getMax();
        }

        if (SummaryColumn.LAST.present(mask)) {
            prevLast += readDelta(stream);
        }

        point.summaryInt64 = new ImmutableSummaryInt64Snapshot(prevCount, prevSum, min, max, prevLast);
    }

    private long readDelta(BitBuf stream) {
        return CodedInputStream.decodeZigZag64(VarintEncoder.readVarintMode64(stream));
    }

    @Override
    protected void resetAdditionalState() {
        mask = 0;
        prevMax = 0;
        prevMin = 0;
        prevCount = 0;
        prevSum = 0;
        prevLast = 0;
    }
}
