package ru.yandex.solomon.codec.compress.summaries;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedOutputStream;

import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.AbstractTimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.CommandEncoder;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.compress.VarintEncoder;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;

import static ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot.EMPTY;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SummaryInt64TimeSeriesOutputStreamV3 extends AbstractTimeSeriesOutputStream {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SummaryInt64TimeSeriesOutputStreamV3.class);

    private int mask = 0;
    private long prevCount = 0;
    private long prevSum = 0;
    private long prevMin = 0;
    private long prevMax = 0;
    private long prevLast = 0;

    public SummaryInt64TimeSeriesOutputStreamV3(BitBuf out, int records) {
        super(out, records);
    }

    public SummaryInt64TimeSeriesOutputStreamV3(SummaryInt64TimeSeriesOutputStreamV3 copy) {
        super(copy);
        this.mask = copy.mask;
        this.prevCount = copy.prevCount;
        this.prevSum = copy.prevSum;
        this.prevMin = copy.prevMin;
        this.prevMax = copy.prevMax;
        this.prevLast = copy.prevLast;
    }

    @Override
    protected void writeValueCommand(BitBuf stream, int columnSet, AggrPointData point) {
        var summary = point.summaryInt64;

        int mask = 0;
        if (summary.getCount() != EMPTY.getCount()) {
            mask |= SummaryColumn.COUNT.getMask();
        }

        if (summary.getSum() != EMPTY.getSum()) {
            mask |= SummaryColumn.SUM.getMask();
        }

        if (summary.getMin() != EMPTY.getMin()) {
            mask |= SummaryColumn.MIN.getMask();
        }

        if (summary.getMax() != EMPTY.getMax()) {
            mask |= SummaryColumn.MAX.getMask();
        }

        if (summary.getLast() != EMPTY.getLast()) {
            mask |= SummaryColumn.LAST.getMask();
        }

        if ((this.mask & mask) != mask) {
            this.mask |= mask;
            CommandEncoder.encodeCommandPrefix(stream, StockpileColumn.ISUMMARY);
            stream.writeBits(this.mask, 5);
        }
    }

    @Override
    protected void writeValue(BitBuf stream, int columnSet, AggrPointData point) {
        SummaryInt64Snapshot summary = point.summaryInt64;

        if (SummaryColumn.COUNT.present(mask)) {
            writeDelta(summary.getCount(), prevCount, stream);
            prevCount = summary.getCount();
        }

        if (SummaryColumn.SUM.present(mask)) {
            writeDelta(summary.getSum(), prevSum, stream);
            prevSum = summary.getSum();
        }

        if (SummaryColumn.MIN.present(mask)) {
            writeDelta(summary.getMin(), prevMin, stream);
            prevMin = summary.getMin();
        }

        if (SummaryColumn.MAX.present(mask)) {
            writeDelta(summary.getMax(), prevMax, stream);
            prevMax = summary.getMax();
        }

        if (SummaryColumn.LAST.present(mask)) {
            writeDelta(summary.getLast(), prevLast, stream);
            prevLast = summary.getLast();
        }
    }

    private void writeDelta(long current, long prev, BitBuf stream) {
        long delta = current - prev;
        VarintEncoder.writeVarintMode64(stream, CodedOutputStream.encodeZigZag64(delta));
    }

    @Override
    protected long memorySelfSize() {
        return SELF_SIZE;
    }

    @Override
    public TimeSeriesOutputStream copy() {
        return new SummaryInt64TimeSeriesOutputStreamV3(this);
    }

    @Override
    protected void dumpAndResetAdditionalState(BitBuf buffer) {
        buffer.writeBits(this.mask, 5);
        mask = 0;

        VarintEncoder.writeVarintMode64(buffer, prevCount);
        prevCount = 0;

        VarintEncoder.writeVarintMode64(buffer, prevSum);
        prevSum = 0;

        VarintEncoder.writeVarintMode64(buffer, prevMin);
        prevMin = 0;

        VarintEncoder.writeVarintMode64(buffer, prevMax);
        prevMax = 0;

        VarintEncoder.writeVarintMode64(buffer, prevLast);
        prevLast = 0;
    }

    @Override
    protected void restoreAdditionalState(BitBuf buffer) {
        mask = buffer.readBitsToInt(5);
        prevCount = VarintEncoder.readVarintMode64(buffer);
        prevSum = VarintEncoder.readVarintMode64(buffer);
        prevMin = VarintEncoder.readVarintMode64(buffer);
        prevMax = VarintEncoder.readVarintMode64(buffer);
        prevLast = VarintEncoder.readVarintMode64(buffer);
    }
}
