package ru.yandex.solomon.codec.serializer;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;

import javax.annotation.Nonnull;

import com.google.protobuf.ByteString;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.algo.arrayList.ArrayListImpl;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Stepan Koltsov
 */
public class HeapStockpileSerializer implements StockpileSerializer {
    private byte[] array;
    private int length = 0;

    public HeapStockpileSerializer() {
        array = Cf.ByteArray.emptyArray();
    }

    public HeapStockpileSerializer(int bytesCapacity) {
        array = new byte[bytesCapacity];
    }

    private int remainingCapacity() {
        return array.length - length;
    }

    private void reserveAdditional(int additional) {
        if (remainingCapacity() < additional) {
            array = ArrayListImpl.reserveAdditionalInArray(array, Cf.ByteArray, length, additional);
        }
    }

    @Override
    public void ensureCapacity(int bytes) {
        reserveAdditional(bytes);
    }

    @Override
    public long memorySizeIncludingSelf() {
        return MemoryCounter.arrayObjectSize(array);
    }

    private class AsOutputStream extends OutputStream {
        @Override
        public void write(int b) throws IOException {
            writeByte(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            writeBytes(b, off, len);
        }
    }

    @Override
    public OutputStream asOutputStream() {
        return new HeapStockpileSerializer.AsOutputStream();
    }

    @Override
    public void writeBoolean(boolean b) {
        writeByte(b ? 1 : 0);
    }

    @Override
    public void writeByte(int b) {
        writeByte((byte) b);
    }

    @Override
    public void writeByte(byte b) {
        reserveAdditional(1);
        array[length] = b;
        ++length;
    }

    @Override
    public void writeFixed64(long value) {
        reserveAdditional(8);
        // trusting JVM to unroll the loop
        for (int i = 0; i < 8; ++i) {
            array[length + i] = (byte) (value >>> (i * 8));
        }
        length += 8;
    }

    @Override
    public void writeFixed32(int value) {
        reserveAdditional(4);
        for (int i = 0; i < 4; ++i) {
            array[length + i] = (byte) (value >>> (i * 8));
        }
        length += 4;
    }

    @Override
    public void writeVarint16(short value) {
        writeVarint32(value);
    }

    @Override
    public void writeVarint32(int value) {
        reserveAdditional(ProtobufVarint.MAX_VARINT_LENGTH);
        length += ProtobufVarint.writeSignedVarint32(array, length, value);
    }

    @Override
    public void writeVarint64(long value) {
        reserveAdditional(ProtobufVarint.MAX_VARINT_LENGTH);
        length += ProtobufVarint.writeSignedVarint64(array, length, value);
    }

    @Override
    public void writeDouble(double v) {
        writeFixed64(Double.doubleToRawLongBits(v));
    }

    private void writeBytesNoReserve(byte[] bytes, int offset, int length) {
        System.arraycopy(bytes, offset, array, this.length, length);
        this.length += length;
    }

    @Override
    public void writeBytes(byte[] bytes, int offset, int length) {
        reserveAdditional(length);
        writeBytesNoReserve(bytes, offset, length);
    }

    @Override
    public void writeBytes(byte[] bytes) {
        writeBytes(bytes, 0, bytes.length);
    }

    @Override
    public int size() {
        return this.length;
    }

    @Nonnull
    public byte[] build() {
        byte[] r = Arrays.copyOf(array, length);
        this.array = Cf.ByteArray.emptyArray();
        this.length = 0;
        return r;
    }

    public byte[] flush() {
        byte[] r = Arrays.copyOf(array, length);
        this.length = 0;
        return r;
    }

    @Nonnull
    public ByteString buildByteString() {
        var result = ByteStringsStockpile.unsafeWrap(array, 0, length);
        this.array = Cf.ByteArray.emptyArray();
        this.length = 0;
        return result;
    }
}
