package ru.yandex.solomon.codec.serializer;

import java.io.OutputStream;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;

import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class NettyStockpileSerializer implements StockpileSerializer {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(NettyStockpileSerializer.class);
    private final ByteBuf buffer;

    public NettyStockpileSerializer(ByteBuf buffer) {
        this.buffer = buffer;
    }

    @Override
    public void ensureCapacity(int bytes) {
        buffer.ensureWritable(bytes);
    }

    @Override
    public OutputStream asOutputStream() {
        return new ByteBufOutputStream(buffer);
    }

    @Override
    public void writeBoolean(boolean b) {
        buffer.writeBoolean(b);
    }

    @Override
    public void writeByte(int b) {
        buffer.writeByte(b);
    }

    @Override
    public void writeByte(byte b) {
        buffer.writeByte(b);
    }

    @Override
    public void writeFixed64(long value) {
        buffer.writeLongLE(value);
    }

    @Override
    public void writeFixed32(int value) {
        buffer.writeIntLE(value);
    }

    @Override
    public void writeVarint16(short value) {
        writeVarint32(value);
    }

    @Override
    public void writeVarint32(int value) {
        buffer.ensureWritable(ProtobufVarint.MAX_VARINT_LENGTH);
        ProtobufVarint.writeUnsignedVarint64(buffer, value);
    }

    @Override
    public void writeVarint64(long value) {
        buffer.ensureWritable(ProtobufVarint.MAX_VARINT_LENGTH);
        ProtobufVarint.writeUnsignedVarint64(buffer, value);
    }

    @Override
    public void writeDouble(double v) {
        writeFixed64(Double.doubleToRawLongBits(v));
    }

    @Override
    public void writeBytes(byte[] bytes, int offset, int length) {
        buffer.writeBytes(bytes, offset, length);
    }

    @Override
    public void writeBytes(byte[] bytes) {
        buffer.writeBytes(bytes);
    }

    @Override
    public int size() {
        return buffer.readableBytes();
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + MemoryCounter.byteBufSize(buffer);
    }

    public ByteBuf getBuffer() {
        return buffer;
    }
}
