package ru.yandex.solomon.codec.serializer;

import java.io.IOException;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import io.netty.buffer.ByteBuf;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ProtobufVarint {

    public static final int MAX_VARINT_LENGTH = 10;
    public static final int MAX_UNSIGNED_INT_LENGTH = 5;

    public static final int MAX_SINGLE_BYTE_VALUE = 127;

    public static int signedVarint32EncodedLength(int value) {
        return CodedOutputStream.computeRawVarint64Size(value);
    }

    public static int unsignedVarint32EncodedLength(int value) {
        return CodedOutputStream.computeRawVarint32Size(value);
    }

    public static int signedVarint64EncodedLength(long value) {
        return CodedOutputStream.computeRawVarint64Size(value);
    }

    public static int unsignedVarint64EncodedLength(long value) {
        return CodedOutputStream.computeRawVarint64Size(value);
    }

    public static class ReadIntResult {
        public final int value;
        public final int length;

        public ReadIntResult(int value, int length) {
            this.value = value;
            this.length = length;
        }
    }

    public static class ReadLongResult {
        public final long value;
        public final int length;

        public ReadLongResult(long value, int length) {
            this.value = value;
            this.length = length;
        }
    }

    public static ReadIntResult readSignedVarint32(byte[] bytes, int offset) {
        // TODO: inline
        CodedInputStream is = CodedInputStream.newInstance(bytes, offset, bytes.length - offset);
        try {
            int v = is.readRawVarint32();
            return new ReadIntResult(v, is.getTotalBytesRead());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static int writeSignedVarint32(byte[] bytes, int offset, int value) {
        return writeSignedVarint64(bytes, offset, value);
    }

    public static ReadIntResult readUnsignedVarint32(byte[] bytes, int offset) {
        // TODO: inline
        CodedInputStream is = CodedInputStream.newInstance(bytes, offset, bytes.length - offset);
        try {
            int v = (int) is.readRawVarint64();
            return new ReadIntResult(v, is.getTotalBytesRead());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static int writeUnsignedVarint32(byte[] bytes, int offset, int value) {
        int pos = 0;
        for (;;) {
            if ((value & ~0x7FL) == 0) {
                bytes[offset + pos] = (byte) value;
                return pos + 1;
            } else {
                bytes[offset + pos] = (byte) (((int)value & 0x7F) | 0x80);
                pos += 1;
                value >>>= 7;
            }
        }
    }

    public static ReadLongResult readSignedVarint64(byte[] bytes, int offset) {
        // TODO: inline
        CodedInputStream is = CodedInputStream.newInstance(bytes, offset, bytes.length - offset);
        try {
            long v = is.readRawVarint64();
            return new ReadLongResult(v, is.getTotalBytesRead());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static ReadLongResult readUnsignedVarint64(byte[] bytes, int offset) {
        // TODO: inline
        CodedInputStream is = CodedInputStream.newInstance(bytes, offset, bytes.length - offset);
        try {
            long v = is.readRawVarint64();
            return new ReadLongResult(v, is.getTotalBytesRead());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static int writeUnsignedVarint64(byte[] bytes, int offset, long value) {
        int pos = 0;
        for (;;) {
            if ((value & ~0x7FL) == 0) {
                bytes[offset + pos] = (byte) value;
                return pos + 1;
            } else {
                bytes[offset + pos] = (byte) (((int)value & 0x7F) | 0x80);
                pos += 1;
                value >>>= 7;
            }
        }
    }

    public static void writeUnsignedVarint64(ByteBuf buffer, long value) {
        for (;;) {
            if ((value & ~0x7FL) == 0) {
                buffer.writeByte((byte) value);
                break;
            } else {
                buffer.writeByte((byte) (((int)value & 0x7F) | 0x80));
                value >>>= 7;
            }
        }
    }

    public static int writeSignedVarint64(byte[] bytes, int offset, long value) {
        return writeUnsignedVarint64(bytes, offset, value);
    }

}
