package ru.yandex.solomon.codec.serializer;

import java.io.IOException;
import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.InvalidProtocolBufferException;
import io.netty.buffer.ByteBuf;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.lang.ShortUtils;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileDeserializer {

    private final CodedInputStream codedInputStream;

    public StockpileDeserializer(CodedInputStream codedInputStream) {
        this.codedInputStream = codedInputStream;
    }

    public StockpileDeserializer(byte[] bytes, int offset, int length) {
        this(CodedInputStream.newInstance(bytes, offset, length));
    }

    public StockpileDeserializer(byte[] bytes) {
        this(bytes, 0, bytes.length);
    }

    public StockpileDeserializer(ByteBuf bytes) {
        this(bytes.array(), bytes.arrayOffset() + bytes.readerIndex(), bytes.readableBytes());
    }

    public StockpileDeserializer(ByteString bytes) {
        this(newCodedInput(bytes));
    }

    public int pos() {
        return codedInputStream.getTotalBytesRead();
    }

    private static CodedInputStream newCodedInput(ByteString bytes) {
        CodedInputStream r = bytes.newCodedInput();
        // https://github.com/google/protobuf/issues/1251
        r.setSizeLimit(bytes.size());
        return r;
    }

    public static StockpileDeserializer concat(byte[]... bytess) {
        if (bytess.length == 0) {
            return new StockpileDeserializer(Cf.ByteArray.emptyArray());
        } else if (bytess.length == 1) {
            return new StockpileDeserializer(bytess[0]);
        } else {
            ByteString[] byteStrings = Arrays.stream(bytess)
                .map(ByteStringsStockpile::unsafeWrap)
                .toArray(ByteString[]::new);
            return concat(byteStrings);
        }
    }

    public static StockpileDeserializer concat(ByteString[] parts) {
        return new StockpileDeserializer(ByteStringsStockpile.concat(parts));
    }

    public CodedInputStream getCodedInputStream() {
        return codedInputStream;
    }

    public boolean atEof() {
        try {
            return codedInputStream.isAtEnd();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public void checkEof() {
        if (!atEof()) {
            throw new IllegalStateException();
        }
    }

    public int remaining() {
        return codedInputStream.getBytesUntilLimit();
    }

    public long readFixed64() {
        try {
            return codedInputStream.readFixed64();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public double readDouble() {
        try {
            return codedInputStream.readDouble();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public int readFixed32() {
        try {
            return codedInputStream.readFixed32();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public short readVarint16() {
        return ShortUtils.toShortExact(readVarint32());
    }

    public int readVarint32() {
        try {
            return codedInputStream.readRawVarint32();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public long readVarint64() {
        try {
            return codedInputStream.readRawVarint64();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public void skipVarint() {
        // TODO: could be better
        try {
            codedInputStream.readRawVarint64();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public byte[] readBytes(int length) {
        try {
            return codedInputStream.readRawBytes(length);
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public void skipBytes(int length) {
        try {
            codedInputStream.skipRawBytes(length);
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public void skipProtobufField(int tag) {
        try {
            codedInputStream.skipField(tag);
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public byte readByte() {
        try {
            return codedInputStream.readRawByte();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public byte[] readBytesLengthDelimited() {
        int length = readVarint32();
        return readBytes(length);
    }

    public String readStringLengthDelimited() {
        try {
            return codedInputStream.readStringRequireUtf8();
        } catch (IOException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public int readByteUnsigned() {
        return Byte.toUnsignedInt(readByte());
    }

    public byte[] readBytesToEof() {
        return readBytes(codedInputStream.getBytesUntilLimit());
    }

    public void skipBytesToEof() {
        // TODO
        readBytesToEof();
    }

    public boolean readBoolean() {
        byte b = readByte();
        if (b == 1) {
            return true;
        } else if (b == 0) {
            return false;
        } else {
            throw new CorruptedBinaryDataRuntimeException("wrong byte in boolean: " + (b & 0xff));
        }
    }

    public int pushLimit(int limit) {
        try {
            return codedInputStream.pushLimit(limit);
        } catch (InvalidProtocolBufferException e) {
            throw new CorruptedBinaryDataRuntimeException(e);
        }
    }

    public void popLimit(int oldLimit) {
        codedInputStream.popLimit(oldLimit);
    }
}
