package ru.yandex.solomon.codec.serializer;

import java.util.EnumMap;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.solomon.codec.UnsupportedFormatRuntimeException;

/**
 * @author Stepan Koltsov
 *
 * https://solomon.yandex-team.ru/?cluster=production&project=solomon&service=stockpile&l.format=*&l.host=cluster&l.sensor=stockpile.host.shard.format&graph=auto&b=1h&e=
 */
@ParametersAreNonnullByDefault
public enum StockpileFormat {
    // <-- add new format here
    DELETED_SHARDS_39(39),
    IDEMPOTENT_WRITE_38(38),
    HISTOGRAM_DENOM_37(37),
    ;

    private static final StockpileFormat[] VALUES = values();

    private final int format;

    public static final StockpileFormat MAX = VALUES[0];
    public static final StockpileFormat MIN = VALUES[VALUES.length - 1];
    public static final StockpileFormat CURRENT = DELETED_SHARDS_39;

    StockpileFormat(int format) {
        int fromName = Integer.parseInt(name().substring(name().lastIndexOf('_') + 1));
        if (fromName != format) {
            throw new IllegalStateException();
        }
        this.format = format;
    }

    public int getFormat() {
        return format;
    }

    public static StockpileFormat byNumber(int number) {
        StockpileFormat format = byNumberOrNull(number);
        if (format == null) {
            throw new UnsupportedFormatRuntimeException("unknown format: " + number);
        }

        return format;
    }

    @Nullable
    public static StockpileFormat byNumberOrNull(int number) {
        for (StockpileFormat stockpileFormat : VALUES) {
            if (stockpileFormat.format == number) {
                return stockpileFormat;
            }
        }

        return null;
    }

    public static StockpileFormat byNumberOrMaxBelow(int number) {
        if (number > MAX.format) {
            return MAX;
        } else {
            return byNumber(number);
        }
    }

    public static StockpileFormat byNumberOrCurrent(int number) {
        if (number >= CURRENT.format) {
            return CURRENT;
        } if (number < MIN.format) {
            return MIN;
        } else {
            return byNumber(number);
        }
    }

    public static <T> EnumMap<StockpileFormat, T> makeMap(Function<StockpileFormat, T> gen) {
        return EnumMapUtils.fill(StockpileFormat.class, gen);
    }

    public boolean isBetweenInclusive(StockpileFormat from, StockpileFormat toIncl) {
        return format >= from.format && format <= toIncl.format;
    }

    public boolean lt(StockpileFormat that) {
        return this.format < that.format;
    }

    public boolean gt(StockpileFormat that) {
        return this.format > that.format;
    }

    public boolean ge(StockpileFormat that) {
        return this.format >= that.format;
    }

    public boolean le(StockpileFormat that) {
        return this.format <= that.format;
    }

    public static int compare(StockpileFormat a, StockpileFormat b) {
        return Integer.compare(a.format, b.format);
    }

    @Nonnull
    public static StockpileFormat max(StockpileFormat a, StockpileFormat b) {
        if (a.format > b.format) {
            return a;
        } else {
            return b;
        }
    }
}
