package ru.yandex.solomon.math;

import java.util.function.LongPredicate;

import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.Timeline;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
 * @author Stepan Koltsov
 */
public class GraphDataAsFunction {

    private final Timeline.Navigator nav;
    private final Interpolate interpolate;
    private final LongArrayView timestamps;
    private final DoubleArrayView values;

    public GraphDataAsFunction(GraphData graphData, Interpolate interpolate) {
        this.interpolate = interpolate;
        nav = graphData.getTimeline().new Navigator();
        values = graphData.getValues();
        timestamps = graphData.getTimestamps();
    }

    public double computeAt(long leftTsMillis, long tsMillis, long rightTsMillis) {
        double interpolatedResult;
        nav.scrollToMillis(tsMillis);
        if (nav.isOnPoint()) {
            interpolatedResult = this.values.at(nav.pos());
        } else if (nav.isBetweenPoints()) {
            interpolatedResult = interpolate.interpolate(
                    timestamps.at(nav.pos()), this.values.at(nav.pos()),
                    timestamps.at(nav.pos() + 1), this.values.at(nav.pos() + 1),
                    tsMillis);
        } else {
            interpolatedResult = Double.NaN;
        }

        if (Double.isNaN(interpolatedResult)) {
            // Possibly there are some points, and we should not return NaN because we failed to interpolate for some reason.
            return computeAverageOnArea(leftTsMillis, tsMillis, rightTsMillis);
        } else {
            return interpolatedResult;
        }
    }

    private double computeAverageOnArea(long leftTsMillis, long tsMillis, long rightTsMillis) {
        if (interpolate == Interpolate.NONE) {
            return Double.NaN;
        }

        long beginTsMillis;
        long endTsMillis;

        boolean includeBegin;
        boolean includeEnd;

        switch (interpolate) {
            case LEFT:
                beginTsMillis = leftTsMillis;
                endTsMillis = tsMillis;

                includeBegin = false;
                includeEnd = true;

                break;
            case RIGHT:
                beginTsMillis = tsMillis;
                endTsMillis = rightTsMillis;

                includeBegin = true;
                includeEnd = false;

                break;
            case LINEAR:
                beginTsMillis = tsMillis - (tsMillis - leftTsMillis) / 2;
                endTsMillis = tsMillis + (rightTsMillis - tsMillis) / 2;

                includeBegin = false;
                includeEnd = true;

                break;
            default:
                throw new IllegalStateException("unknown interpolate: " + interpolate);
        }
        LongPredicate isAfterAreaEnd = ts -> ts > endTsMillis || (ts == endTsMillis && !(includeEnd || (includeBegin && ts == beginTsMillis)));

        double sumPointsInArea = 0;
        int countPointsInArea = 0;

        nav.scrollToMillis(beginTsMillis);

        while (nav.pos() < timestamps.length()) {
            long ts = timestamps.at(nav.pos());

            if (isAfterAreaEnd.test(ts)) {
                break;
            }

            if (ts > beginTsMillis || (includeBegin && ts == beginTsMillis)) {
                double value = values.at(nav.pos());
                if (!Double.isNaN(value)) {
                    sumPointsInArea += value;
                    countPointsInArea++;
                }
            }

            int oldPos = nav.pos();
            if (oldPos == timestamps.length() - 1) {
                break;
            }
            nav.scrollToMillis(Math.min(endTsMillis, timestamps.at(nav.pos() + 1)));
            if (oldPos == nav.pos()) {
                break;
            }
        }
        if (countPointsInArea != 0) {
            return sumPointsInArea / countPointsInArea;
        } else {
            return Double.NaN;
        }
    }

}
