package ru.yandex.solomon.math;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.SortedOrCheck;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class GraphDataIntegrate {

    @Nonnull
    public static GraphData trapezoidalIntegrateC(GraphData graphData, double c) {
        if (graphData.length() < 2) {
            return GraphData.empty;
        }
        long[] timestamps = graphData.getTimestamps().drop(1).copyToArray();
        double[] values = new double[timestamps.length];
        for (int i = 0; i < timestamps.length; ++i) {
            double d;
            if (Double.isNaN(graphData.getValues().at(i + 1)) || Double.isNaN(graphData.getValues().at(i))) {
                d = 0;
            } else {
                d = (graphData.getValues().at(i + 1) + graphData.getValues().at(i)) * (graphData.getTimestamps().at(i + 1) - graphData.getTimestamps().at(i)) / 2000;
            }
            double prev = i > 0 ? values[i - 1] : c;
            values[i] = prev + d;
        }
        return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
    }

    /**
     * This rule is consistent with derivative/rate
     */
    @Nonnull
    public static GraphData rightRectangleIntegrate(GraphData graphData) {
        if (graphData.length() == 0) {
            return GraphData.empty;
        }
        long[] timestamps = graphData.getTimestamps().copyToArray();
        double[] values = new double[timestamps.length];
        values[0] = 0;
        for (int i = 1; i < timestamps.length; ++i) {
            double delta;
            double integrand = graphData.getValues().at(i);
            if (Double.isNaN(integrand)) {
                delta = 0;
            } else {
                delta = integrand * (timestamps[i] - timestamps[i - 1]) / 1000;
            }
            values[i] = values[i - 1] + delta;
        }
        return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
    }

}
