package ru.yandex.solomon.math;

import java.time.Duration;

import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.GraphDataArrayList;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
 * @author Vladimir Gordiychuk
 */
public final class GraphDataMath {
    private GraphDataMath() {
    }

    public static GraphData tail(GraphData graphData, int countPoints) {
        int dropCount = graphData.length() - countPoints;
        if (dropCount <= 0)
            return graphData;

        return graphData.slice(dropCount, graphData.length());
    }

    public static GraphData tail(GraphData graphData, Duration duration) {
        if (graphData.isEmpty()) {
            return graphData;
        }

        LongArrayView timestamps = graphData.getTimestamps();
        long dropBefore = timestamps.last() - duration.toMillis();
        int fromIndex = 0;
        for (; fromIndex < timestamps.length(); fromIndex++) {
            if (timestamps.at(fromIndex) >= dropBefore) {
                break;
            }
        }

        return graphData.slice(fromIndex, graphData.length());
    }

    public static GraphData dropTail(GraphData graphData, int countPoints) {
        int keepCount = graphData.length() - countPoints;
        if (keepCount <= 0)
            return GraphData.empty;

        return graphData.slice(0, keepCount);
    }

    public static GraphData dropTail(GraphData graphData, Duration duration) {
        if (graphData.isEmpty()) {
            return graphData;
        }

        LongArrayView timestamps = graphData.getTimestamps();
        long keepBefore = timestamps.last() - duration.toMillis();
        int toIndex = 0;
        for (; toIndex < timestamps.length(); toIndex++) {
            if (timestamps.at(toIndex) >= keepBefore) {
                break;
            }
        }

        return graphData.slice(0, toIndex);
    }

    public static GraphData head(GraphData graphData, int countPoints) {
        if (graphData.length() <= countPoints) {
            return graphData;
        }

        return graphData.slice(0, countPoints);
    }

    public static GraphData head(GraphData graphData, Duration duration) {
        if (graphData.isEmpty()) {
            return graphData;
        }

        LongArrayView timestamps = graphData.getTimestamps();
        long dropAfter = timestamps.first() + duration.toMillis();
        int toIndex = 0;
        for (; toIndex < timestamps.length(); toIndex++) {
            if (timestamps.at(toIndex) > dropAfter) {
                break;
            }
        }

        return graphData.slice(0, toIndex);
    }

    public static GraphData dropHead(GraphData graphData, int countPoints) {
        if (graphData.length() <= countPoints) {
            return GraphData.empty;
        }

        return graphData.slice(countPoints, graphData.length());
    }

    public static GraphData dropHead(GraphData graphData, Duration duration) {
        if (graphData.isEmpty()) {
            return graphData;
        }

        LongArrayView timestamps = graphData.getTimestamps();
        long keepAfter = timestamps.first() + duration.toMillis();
        int fromIndex = 0;
        for (; fromIndex < timestamps.length(); fromIndex++) {
            if (timestamps.at(fromIndex) > keepAfter) {
                break;
            }
        }

        return graphData.slice(fromIndex, graphData.length());
    }

    public static GraphData dropIf(GraphData condition, GraphData graphData) {
        if (graphData.isEmpty() || condition.isEmpty()) {
            return graphData;
        }

        LongArrayView condTimestamps = condition.getTimestamps();
        DoubleArrayView condValues = condition.getValues();

        LongArrayView timestamps = graphData.getTimestamps();
        DoubleArrayView values = graphData.getValues();

        GraphDataArrayList builder = new GraphDataArrayList();

        boolean takePoint = true;
        int condIndex = 0;

        for (int index = 0; index < timestamps.length(); index++) {
            long pointTime = timestamps.at(index);
            while (condIndex < condTimestamps.length() && condTimestamps.at(condIndex) <= pointTime) {
                double condValue = condValues.at(condIndex);
                condIndex++;
                if (Double.isNaN(condValue)) {
                    continue;
                }
                takePoint = condValue == 0;
            }
            if (takePoint) {
                builder.add(pointTime, values.at(index));
            }
        }

        return builder.buildGraphData();
    }

    public static GraphData takeIf(GraphData condition, GraphData graphData) {
        if (graphData.isEmpty() || condition.isEmpty()) {
            return GraphData.empty;
        }

        LongArrayView condTimestamps = condition.getTimestamps();
        DoubleArrayView condValues = condition.getValues();

        LongArrayView timestamps = graphData.getTimestamps();
        DoubleArrayView values = graphData.getValues();

        GraphDataArrayList builder = new GraphDataArrayList();

        boolean takePoint = false;
        int condIndex = 0;

        for (int index = 0; index < timestamps.length(); index++) {
            long pointTime = timestamps.at(index);
            while (condIndex < condTimestamps.length() && condTimestamps.at(condIndex) <= pointTime) {
                double condValue = condValues.at(condIndex);
                condIndex++;
                if (Double.isNaN(condValue)) {
                    continue;
                }
                takePoint = condValue != 0;
            }
            if (takePoint) {
                builder.add(pointTime, values.at(index));
            }
        }

        return builder.buildGraphData();
    }

    public static DoubleArrayView getTimestampsAsEpochSeconds(GraphData graphData) {
        LongArrayView timestamps = graphData.getTimestamps();
        DoubleArrayView values = graphData.getValues();

        double[] builder = new double[timestamps.length()];
        int putIndex = 0;

        for (int index = 0; index < timestamps.length(); index++) {
            long pointTime = timestamps.at(index);
            if (!Double.isNaN(values.at(index))) {
                builder[putIndex] = pointTime * 1e-3;
                putIndex++;
            }
        }

        return new DoubleArrayView(builder, 0, putIndex);
    }
}
