package ru.yandex.solomon.math;

import javax.annotation.Nullable;

/**
 * @author Stepan Koltsov
 */
public enum Interpolate {
    LINEAR,
    LEFT,
    RIGHT,
    NONE,
    ;

    public static final Interpolate DEFAULT = LINEAR;

    private final String parameterValue;

    Interpolate() {
        this.parameterValue = name().toLowerCase();
    }

    public String getParameterValue() {
        return parameterValue;
    }

    public static Interpolate parse(@Nullable String parameterValue) {
        if (parameterValue == null || parameterValue.isEmpty()) {
            return DEFAULT;
        }
        for (Interpolate interpolate : values()) {
            if (interpolate.parameterValue.equals(parameterValue)) {
                return interpolate;
            }
        }
        throw new IllegalArgumentException("unknown value: " + parameterValue);
    }

    public double interpolate(long ax, double ay, long bx, double by, long cx) {
        switch (this) {
            case LINEAR: return linear(ax, ay, bx, by, cx);
            case LEFT: return !Double.isNaN(by) ? ay : Double.NaN;
            case RIGHT: return !Double.isNaN(ay) ? by : Double.NaN;
            case NONE: return Double.NaN;
            default:
                throw new IllegalStateException("unknown interpolate: " + this);
        }
    }

    public static double linear(long ax, double ay, long bx, double by, long cx) {
        return 1.0 * ((bx - cx) * ay + (cx - ax) * by) / (bx - ax);
    }
}
