package ru.yandex.solomon.math.doubles;

import java.util.function.IntToLongFunction;
import java.util.stream.DoubleStream;

import org.apache.commons.math3.stat.descriptive.rank.Percentile;

import ru.yandex.solomon.model.timeseries.SortedOrCheck;
import ru.yandex.solomon.model.timeseries.Timeline;
import ru.yandex.solomon.model.timeseries.view.DoubleTimeSeriesView;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;

/**
 * @author Vladimir Gordiychuk
 */
public class DoubleAggregateFunctions {

    public static double max(DoubleTimeSeriesView timeSeries) {
        return max(timeSeries.streamValues());
    }

    public static double max(DoubleArrayView doubleArrayView) {
        return max(doubleArrayView.stream());
    }

    public static double max(DoubleStream stream) {
        return skipNan(stream).max().orElse(Double.NaN);
    }

    public static double min(DoubleTimeSeriesView timeSeries) {
        return min(timeSeries.streamValues());
    }

    public static double min(DoubleArrayView doubleArrayView) {
        return min(doubleArrayView.stream());
    }

    public static double min(DoubleStream stream) {
        return skipNan(stream).min().orElse(Double.NaN);
    }

    public static double sum(DoubleTimeSeriesView timeSeries) {
        return sum(timeSeries.streamValues());
    }

    public static double sum(DoubleArrayView doubleArrayView) {
        return sum(doubleArrayView.stream());
    }

    public static double sum(DoubleStream stream) {
        return skipNan(stream).sum();
    }

    public static double last(DoubleTimeSeriesView timeSeries) {
        if (timeSeries.isEmpty()) {
            return Double.NaN;
        }

        return timeSeries.getValue(timeSeries.length() - 1);
    }

    public static double last(DoubleArrayView doubleArrayView) {
        if (doubleArrayView.isEmpty()) {
            return Double.NaN;
        }

        return doubleArrayView.at(doubleArrayView.length() - 1);
    }

    public static double lastSkipNan(DoubleArrayView doubleArrayView) {
        for (int index = doubleArrayView.length() - 1; index >= 0; --index) {
            double value = doubleArrayView.at(index);
            if (!Double.isNaN(value)) {
                return value;
            }
        }

        return Double.NaN;
    }

    public static double weightedAverage(DoubleTimeSeriesView timeSeries) {
        if (timeSeries.isEmpty()) {
            return Double.NaN;
        }

        if (timeSeries.length() == 1) {
            return timeSeries.getValue(0);
        }

        IntToLongFunction weight = new Timeline(timeSeries.getTimeView(), SortedOrCheck.SORTED_UNIQUE)::lengthAt;

        long weightSum = 0;
        double sum = 0;
        for (int i = 0; i < timeSeries.length(); ++i) {
            double value = timeSeries.getValue(i);
            if (Double.isNaN(value)) {
                continue;
            }

            long weightAt = weight.applyAsLong(i);
            if (weightAt == 0) {
                continue;
            }

            weightSum += weightAt;
            sum += value * weightAt;
        }

        return sum / weightSum;
    }

    public static double average(DoubleArrayView doubleArrayView) {
        return average(doubleArrayView.stream());
    }

    private static double average(DoubleStream stream) {
        return skipNan(stream).average().orElse(Double.NaN);
    }

    public static double count(DoubleTimeSeriesView timeSeries) {
        return count(timeSeries.streamValues());
    }

    public static double count(DoubleArrayView doubleArrayView) {
        return count(doubleArrayView.stream());
    }

    public static double count(DoubleStream stream) {
        return skipNan(stream).count();
    }

    public static double standardDeviation(DoubleArrayView doubleArrayView) {
        double[] array = skipNan(doubleArrayView.stream()).toArray();
        if (array.length <= 1) {
            return Double.NaN;
        }

        double avg = average(doubleArrayView);
        return Math.sqrt(DoubleStream.of(array).map(x -> Math.pow(x - avg, 2)).sum() / (array.length - 1));
    }

    public static double median(DoubleArrayView doubleArrayView) {
        return percentile(doubleArrayView, 50);
    }

    public static double percentile(DoubleArrayView doubleArrayView, double level) {
        double[] array = skipNan(doubleArrayView.stream()).toArray();

        Percentile percentile = new Percentile();
        return percentile.evaluate(array, level);
    }

    public static double interquartileRange(DoubleArrayView param) {
        double[] array = skipNan(param.stream()).toArray();

        Percentile percentile = new Percentile();
        percentile.setData(array);
        return percentile.evaluate(75) - percentile.evaluate(25);
    }

    private static class RightRectangleWeight implements IntToLongFunction {
        private final Timeline timeline;

        RightRectangleWeight(Timeline timeline) {
            this.timeline = timeline;
        }

        @Override
        public long applyAsLong(int i) {
            if (i == 0) {
                return 0;
            }
            return timeline.getPointMillisAt(i) - timeline.getPointMillisAt(i - 1);
        }
    }

    public static double integrate(DoubleTimeSeriesView timeSeries) {
        IntToLongFunction weight = new RightRectangleWeight(new Timeline(timeSeries.getTimeView(), SortedOrCheck.SORTED_UNIQUE));
        return integrate(timeSeries, weight);
    }

    public static double integrate(DoubleTimeSeriesView view, IntToLongFunction weight) {
        double sum = 0;
        for (int i = 0; i < view.length(); ++i) {
            double value = view.getValue(i);
            if (Double.isNaN(value)) {
                continue;
            }

            long weightAt = weight.applyAsLong(i);
            if (weightAt == 0) {
                continue;
            }

            sum += value * weightAt / 1000;
        }

        return sum;
    }

    private static DoubleStream skipNan(DoubleStream stream) {
        return stream.filter(d -> !Double.isNaN(d));
    }
}
