package ru.yandex.solomon.math.doubles;

import java.util.function.DoubleUnaryOperator;

import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;

/**
 * @author Vladimir Gordiychuk
 */
public class DoubleMath {

    public static double heaviside(double value) {
        return (value > 0) ? 1.0 : (Double.isNaN(value) ? Double.NaN : 0.0);
    }

    public static double ramp(double value) {
        return value > 0 ? value : Double.isNaN(value) ? Double.NaN : 0;
    }

    public static double[] map(double[] value, DoubleUnaryOperator fn) {
        var result = new double[value.length];
        for (int index = 0; index < value.length; index++) {
            result[index] = fn.applyAsDouble(value[index]);
        }
        return result;
    }

    public static AggrGraphDataArrayListOrView map(MetricType dataType, AggrGraphDataArrayListOrView source, DoubleUnaryOperator fn) {
        if (source.isEmpty()) {
            return source;
        }

        var point = RecyclableAggrPoint.newInstance();
        var writePoint = RecyclableAggrPoint.newInstance();
        writePoint.reset();
        try {
            var result = new AggrGraphDataArrayList(StockpileColumns.minColumnSet(MetricType.DGAUGE), source.length());
            var it = MetricTypeTransfers.of(dataType, MetricType.DGAUGE, source.iterator());
            while (it.next(point)) {
                writePoint.setTsMillis(point.getTsMillis());
                writePoint.setValue(fn.applyAsDouble(point.getValueDivided()));
                result.addRecord(writePoint);
            }
            return result;
        } finally {
            point.recycle();
            writePoint.recycle();
        }
    }

    public static double trunc(double v) {
        return (v > 0) ? Math.floor(v) : Math.ceil(v);
    }

    public static double fract(double v) {
        return Double.isInfinite(v) ? 0 : v % 1d;
    }
}
