package ru.yandex.solomon.math.operation;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import ru.yandex.solomon.math.operation.map.MapOperation;
import ru.yandex.solomon.math.operation.map.OperationAggregationSummary;
import ru.yandex.solomon.math.operation.map.OperationCast;
import ru.yandex.solomon.math.operation.map.OperationDownsampling;
import ru.yandex.solomon.math.operation.reduce.OperationCombine;
import ru.yandex.solomon.math.operation.reduce.OperationTop;
import ru.yandex.solomon.math.operation.reduce.ReduceOperation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class OperationsPipeline<Key> {
    public final OperationsPipeline<Key> downsampling(Interval interval, ru.yandex.solomon.math.protobuf.OperationDownsampling opts) {
        return thenMap(new OperationDownsampling<>(interval, opts));
    }

    public final OperationsPipeline<Key> dropTimeSeries(OperationDropTimeSeries opts) {
        return thenMap(Metric::dropTimeseries);
    }

    public final OperationsPipeline<Key> top(ru.yandex.solomon.math.protobuf.OperationTop opts) {
        return thenReduce(new OperationTop<>(opts));
    }

    public final OperationsPipeline<Key> combine(ru.yandex.solomon.math.protobuf.OperationCombine opts) {
        return thenReduce(new OperationCombine<>(opts));
    }

    public final OperationsPipeline<Key> summary(ru.yandex.solomon.math.protobuf.OperationAggregationSummary opts) {
        return thenMap(new OperationAggregationSummary<>(opts));
    }

    public final OperationsPipeline<Key> cast(ru.yandex.solomon.math.protobuf.OperationCast opts) {
        return thenMap(new OperationCast<>(opts));
    }

    public abstract <T> CompletableFuture<List<T>> collect(Function<Metric<Key>, T> mapping);

    public final OperationsPipeline<Key> apply(Interval interval, Operation opts) {
        switch (opts.getTypeCase()) {
            case TYPE_NOT_SET:
                return this;
            case DROP_TIMESERIES:
                return dropTimeSeries(opts.getDropTimeseries());
            case TOP:
                return top(opts.getTop());
            case DOWNSAMPLING:
                return downsampling(interval, opts.getDownsampling());
            case COMBINE:
                return combine(opts.getCombine());
            case SUMMARY:
                return summary(opts.getSummary());
            case CAST:
                return cast(opts.getCast());
            default:
                throw new UnsupportedOperationException("unknown operation type: " + opts.getTypeCase());
        }
    }

    public final OperationsPipeline<Key> apply(Interval interval, List<Operation> operations) {
        OperationsPipeline<Key> result = this;
        for (Operation op : operations) {
            result = result.apply(interval, op);
        }
        return result;
    }

    abstract OperationsPipeline<Key> thenMap(MapOperation<Key> map);
    abstract OperationsPipeline<Key> thenReduce(ReduceOperation<Key> reduce);

}
