package ru.yandex.solomon.math.operation.map;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollector;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DownsamplingIterator extends AggrGraphDataListIterator {
    /**
     * One point every 5 min for 1 year interval
     */
    private static final int POINTS_MAX_COUNT = (int) TimeUnit.DAYS.toMinutes(360) / 5;

    private final AggrGraphDataListIterator source;
    private final Interval interval;
    private final long stepMillis;
    private final PointValueCollector collector;
    private final OperationDownsampling.FillOption fillOpt;
    private final boolean ignoreMinStepMillis;

    private long prevTsMillis;
    private final AggrPoint prev;
    private final AggrPoint lastAppended;
    private final AggrPoint buf;
    private long windowStartMillis;
    private long windowEndMillis;
    private boolean empty = true;
    private int points;

    public DownsamplingIterator(
            int mask,
            AggrGraphDataListIterator source,
            Interval interval,
            long stepMillis,
            PointValueCollector collector,
            OperationDownsampling.FillOption fillOpt,
            boolean ignoreMinStepMillis)
    {
        super(mask);
        this.source = source;
        if (stepMillis <= 0) {
            throw new IllegalArgumentException("StepMillis should be positive, but was " + stepMillis);
        }

        this.stepMillis = stepMillis;
        this.interval = Interval.truncate(interval, stepMillis);
        this.collector = collector;
        this.fillOpt = fillOpt;
        this.ignoreMinStepMillis = ignoreMinStepMillis;

        this.windowStartMillis = this.interval.getBeginMillis();
        this.windowEndMillis = this.windowStartMillis + stepMillis;
        this.buf = new AggrPoint(columnSetMask());
        this.lastAppended = new AggrPoint(columnSetMask());
        this.prev = new AggrPoint(columnSetMask());

        int totalCountPoints = estimatePointsCount();
        if (totalCountPoints > POINTS_MAX_COUNT) {
            throw new IllegalArgumentException("Requested step " +
                    stepMillis +
                    "ms to small for interval " +
                    interval +
                    " and will produce " +
                    totalCountPoints +
                    " points"
            );
        }
    }

    @Override
    public int estimatePointsCount() {
        return Math.toIntExact(interval.duration().toMillis() / stepMillis);
    }

    @Override
    public boolean next(AggrPoint target) {
        if (loopGaps(target)) {
            return true;
        }

        while (fetchNext(buf)) {
            if (buf.tsMillis < windowEndMillis) {
                append(buf);
                continue;
            }

            if (compute(target)) {
                return true;
            }

            if (loopGaps(target)) {
                return true;
            }
        }

        if (!empty) {
           return compute(target);
        } else if (points == 0) {
            return false;
        }

        if (windowStartMillis >= interval.getEndMillis()) {
            return false;
        }

        return compute(target);
    }

    private boolean fetchNext(AggrPoint tmp) {
        if (!source.next(tmp)) {
            tmp.tsMillis = TsColumn.DEFAULT_VALUE;
            return false;
        }
        return true;
    }

    private boolean loopGaps(AggrPoint target) {
        while (buf.getTsMillis() >= windowEndMillis) {
            if (compute(target)) {
                return true;
            }
        }

        if (buf.getTsMillis() != TsColumn.DEFAULT_VALUE) {
            append(buf);
        }

        return false;
    }

    private void append(AggrPoint target) {
        empty = false;
        collector.append(target);
        target.copyTo(lastAppended);
        prevTsMillis = target.tsMillis;
        target.tsMillis = TsColumn.DEFAULT_VALUE;
    }

    private boolean compute(AggrPoint target) {
        try {
            if (collector.compute(target)) {
                collector.reset();
                target.setTsMillis(windowStartMillis);
                target.copyTo(prev);
                points++;
                return true;
            }

            return fillGap(target);
        } finally {
            this.empty = true;
            this.windowStartMillis += stepMillis;
            this.windowEndMillis += stepMillis;
        }
    }

    private boolean fillGap(AggrPoint target) {
        if (ignoreGapBecauseOfStepMillis()) {
            return false;
        }

        switch (fillOpt) {
            case NULL:
                target.setTsMillis(windowStartMillis);
                target.copyTo(prev);
                prevTsMillis = windowStartMillis;
                return true;
            case NONE:
                return false;
            case PREVIOUS:
                target.setTsMillis(windowStartMillis);
                if (prev.getTsMillis() > 0) {
                    prev.setTsMillis(windowStartMillis);
                    prev.count = 0;
                    prev.copyTo(target);
                    prevTsMillis = windowStartMillis;
                }

                return true;
            default:
                throw new UnsupportedOperationException("Unsupported fill opts: " + fillOpt);
        }
    }

    private boolean ignoreGapBecauseOfStepMillis() {
        if (ignoreMinStepMillis) {
            return false;
        }

        boolean noPointsYet = lastAppended.getTsMillis() == TsColumn.DEFAULT_VALUE;

        if (noPointsYet) {
            return false;
        }

        if (windowEndMillis - prevTsMillis <= lastAppended.getStepMillis()) {
            return true;
        }

        return false;
    }
}
