package ru.yandex.solomon.math.operation.reduce;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.EmptyAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollector;
import ru.yandex.solomon.model.timeseries.iterator.AggrPointCursor;
import ru.yandex.solomon.model.timeseries.iterator.GenericCombineIterator;
import ru.yandex.solomon.model.timeseries.iterator.GenericPointCollector;

/**
 * @author Vladimir Gordiychuk
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
public class CombineIterator extends AggrGraphDataListIterator {
    private final GenericCombineIterator<AggrPointCursor, AggrPoint> backend;

    private CombineIterator(int columnSetMask, List<AggrPointCursor> cursors, PointValueCollector collector) {
        super(columnSetMask);
        this.backend = new GenericCombineIterator<>(cursors, new PointValueCollectorAdapter(collector));
    }

    public static AggrGraphDataListIterator of(int mask, List<AggrGraphDataListIterator> iterators, PointValueCollector collector) {
        if (iterators.isEmpty()) {
            return new EmptyAggrGraphDataIterator(mask);
        }

        List<AggrPointCursor> cursors = iterators.stream()
                .map(AggrPointCursor::new)
                .collect(Collectors.toList());

        return new CombineIterator(mask, cursors, collector);
    }

    @Override
    public int estimatePointsCount() {
        return backend.getCursors().stream()
                .mapToInt(c -> c.getIterator().estimatePointsCount())
                .max()
                .orElse(-1);
    }

    @Override
    public boolean next(AggrPoint target) {
        return backend.next(target);
    }

    private static class PointValueCollectorAdapter implements GenericPointCollector<AggrPointCursor, AggrPoint> {
        private final PointValueCollector collector;

        private PointValueCollectorAdapter(PointValueCollector collector) {
            this.collector = collector;
        }

        @Override
        public void reset() {
            collector.reset();
        }

        @Override
        public void append(int cursorIndex, AggrPointCursor cursor) {
            collector.append(cursor.getPoint());
        }

        @Override
        public void compute(long nextTsMillis, AggrPoint target) {
            collector.compute(target);
            target.setTsMillis(nextTsMillis);
        }
    }

}
