package ru.yandex.solomon.math.operation.reduce;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.operation.comparation.PointValueComparators;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollector;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollectors;

/**
 * @author Vladimir Gordiychuk
 */
public class OperationTop<Key> implements ReduceOperation<Key> {
    private final ru.yandex.solomon.math.protobuf.OperationTop opts;

    public OperationTop(ru.yandex.solomon.math.protobuf.OperationTop opts) {
        this.opts = opts;
    }

    public List<Metric<Key>> apply(List<Metric<Key>> metrics) {
        if (opts.getLimit() == 0) {
            return metrics;
        }

        if (metrics.size() <= opts.getLimit()) {
            return metrics;
        }

        return metrics.stream()
                .map(metric -> Aggregated.of(metric, opts.getTimeAggregation()))
                .sorted(comparator())
                .limit(opts.getLimit())
                .map(aggregated -> aggregated.result)
                .collect(Collectors.toList());
    }

    private Comparator<Aggregated<Key>> comparator() {
        Comparator<Aggregated<Key>> comparator = Aggregated::compareTo;
        if (!opts.getAsc()) {
            comparator = comparator.reversed();
        }

        return Comparator.comparing(Aggregated<Key>::isEmpty)
                .thenComparing(comparator);
    }

    @ParametersAreNonnullByDefault
    private static class Aggregated<Key> implements Comparable<Aggregated<Key>> {
        private final MetricType type;
        private final AggrPoint aggregated;
        private final boolean empty;
        private Metric<Key> result;
        private Comparator<AggrPoint> comparator;

        Aggregated(MetricType type, AggrPoint aggregated, boolean empty, Metric<Key> result) {
            this.type = type;
            this.aggregated = aggregated;
            this.empty = empty;
            this.result = result;
            this.comparator = PointValueComparators.comparator(type);
        }

        static <Key> Aggregated<Key> of(Metric<Key> source, Aggregation aggregation) {
            AggrGraphDataIterable timeseries = source.getTimeseries();
            if (timeseries == null) {
                throw new IllegalArgumentException("Absent timeseries for combine: " + source);
            }

            PointValueCollector collector = PointValueCollectors.of(source.getType(), aggregation);

            AggrPoint point = new AggrPoint(timeseries.columnSetMask());
            AggrGraphDataListIterator it = timeseries.iterator();
            while (it.next(point)) {
                collector.append(point);
            }

            boolean empty = !collector.compute(point);
            return new Aggregated<>(source.getType(), point, empty, source);
        }

        @Override
        public int compareTo(Aggregated that) {
            int compare = this.type.compareTo(that.type);
            if (compare != 0) {
                return compare;
            }

            return comparator.compare(this.aggregated, that.aggregated);
        }

        public boolean isEmpty() {
            return empty;
        }
    }
}
