package ru.yandex.solomon.math.stat;

import java.time.Duration;
import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class CalendarSplitter {

    private final long offsetMillis;
    private final int intervalsPerDay;
    private final long intervalMillis;
    private final DailyProfile dailyProfile;

    private final static int MILLIS_PER_DAY = 86400_000;

    public CalendarSplitter(int intervalsPerDay, DailyProfile dailyProfile) {
        this(intervalsPerDay, dailyProfile, Duration.ZERO);
    }

    public CalendarSplitter(int intervalsPerDay, DailyProfile dailyProfile, Duration timeZone) {
        if (intervalsPerDay > 24*60)
            throw new IllegalArgumentException("Bucket size is too fine, minimal is 1min");
        this.offsetMillis = timeZone.toMillis();
        this.intervalsPerDay = intervalsPerDay;
        this.dailyProfile = dailyProfile;
        this.intervalMillis = MILLIS_PER_DAY / intervalsPerDay;
    }

    public int dayBucketOf(long timestampMillis) {
        long localEpochMillis = timestampMillis + offsetMillis;
        long localEpochDay = Math.floorDiv(localEpochMillis, MILLIS_PER_DAY);
        LocalDate date = LocalDate.ofEpochDay(localEpochDay);

        return dailyProfile.getDayBucket(date);
    }

    public int timeBucketOf(long timestampMillis) {
        long localEpochMillis = timestampMillis + offsetMillis;
        long millisOfDay = Math.floorMod(localEpochMillis, MILLIS_PER_DAY);

        return (int)Math.floorDiv(millisOfDay * intervalsPerDay, MILLIS_PER_DAY);
    }

    public long getBucketWidthMillis() {
        return intervalMillis;
    }

    public int bucketOf(long timestampMillis) {
        long localEpochMillis = timestampMillis + offsetMillis;
        long localEpochDay = Math.floorDiv(localEpochMillis, MILLIS_PER_DAY);
        long millisOfDay = Math.floorMod(localEpochMillis, MILLIS_PER_DAY);
        LocalDate date = LocalDate.ofEpochDay(localEpochDay);

        int dayBucket = dailyProfile.getDayBucket(date);
        int timeBucket = (int)Math.floorDiv(millisOfDay * intervalsPerDay, MILLIS_PER_DAY);
        return dayBucket * intervalsPerDay + timeBucket;
    }

    public static class BucketWithStart {
        final int bucketNumber;
        final long startMillis;
        public BucketWithStart(int bucketNumber, long startMillis) {
            this.bucketNumber = bucketNumber;
            this.startMillis = startMillis;
        }
    }

    public BucketWithStart bucketWithStartOf(long timestampMillis) {
        long localEpochMillis = timestampMillis + offsetMillis;
        long localEpochDay = Math.floorDiv(localEpochMillis, MILLIS_PER_DAY);
        long millisOfDay = Math.floorMod(localEpochMillis, MILLIS_PER_DAY);
        LocalDate date = LocalDate.ofEpochDay(localEpochDay);

        int dayBucket = dailyProfile.getDayBucket(date);
        int timeBucket = (int)Math.floorDiv(millisOfDay * intervalsPerDay, MILLIS_PER_DAY);
        int bucketNumber = dayBucket * intervalsPerDay + timeBucket;
        long startMillis = localEpochDay * MILLIS_PER_DAY + timeBucket * intervalMillis - offsetMillis;
        return new BucketWithStart(bucketNumber, startMillis);
    }

    public static class BucketPair {
        public final int left, right;
        public final double leftWeight, rightWeight;
        public BucketPair(int left, int right, double rightWeight) {
            this.left = left;
            this.right = right;
            this.leftWeight = 1.0 - rightWeight;
            this.rightWeight = rightWeight;
        }
    }

    public BucketPair bucketPairOf(long timestampMillis) {
        long halfIntervalMillis = intervalMillis / 2;
        long leftLookupMillis = timestampMillis - halfIntervalMillis;
        long rightLookupMillis = leftLookupMillis + intervalMillis;

        BucketWithStart left = bucketWithStartOf(leftLookupMillis);
        int right = bucketOf(rightLookupMillis);

        double rightWeight = 1.0 * (leftLookupMillis - left.startMillis) / intervalMillis;

        return new BucketPair(left.bucketNumber, right, rightWeight);
    }

    public int bucketCount() {
        return intervalsPerDay * dailyProfile.bucketCount();
    }
}
