package ru.yandex.solomon.math.stat;

import java.time.LocalDate;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;


/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public enum DailyProfile {
    ANYDAY {
        @Override
        public int getDayBucket(LocalDate date) {
            return 0;
        }
        @Override
        public int bucketCount() {
            return 1;
        }
        @Override
        public String getName() {
            return "anyday";
        }
    },
    DAILY {
        @Override
        public int getDayBucket(LocalDate date) {
            return date.getDayOfWeek().getValue() - 1;
        }
        @Override
        public int bucketCount() {
            return 7;
        }
        @Override
        public String getName() {
            return "daily";
        }
    },
    WORK {
        @Override
        public int getDayBucket(LocalDate date) {
            int dow = date.getDayOfWeek().getValue();
            return (dow <= 5) ? 0 : 1;
        }
        @Override
        public int bucketCount() {
            return 2;
        }
        @Override
        public String getName() {
            return "work";
        }
    },
    SUNSAT {
        @Override
        public int getDayBucket(LocalDate date) {
            int dow = date.getDayOfWeek().getValue();
            return (dow <= 5) ? 0 : dow - 5;
        }
        @Override
        public int bucketCount() {
            return 3;
        }
        @Override
        public String getName() {
            return "sunsat";
        }
    },
    WEEKEND {
        @Override
        public int getDayBucket(LocalDate date) {
            int dow = date.getDayOfWeek().getValue();
            return (dow <= 4) ? 0 : dow - 4;
        }
        @Override
        public int bucketCount() {
            return 4;
        }
        @Override
        public String getName() {
            return "weekend";
        }
    };

    public static Optional<DailyProfile> byName(String getName) {
        for (DailyProfile prof : values()) {
            if (prof.getName().equals(getName)) {
                return Optional.of(prof);
            }
        }
        return Optional.empty();
    }

    public abstract int getDayBucket(LocalDate date);
    public abstract int bucketCount();
    public abstract String getName();
}
